var PersistentStorage = require('./persistent_storage');
var _ = require('./utils');

var keys = { data: 'data', protocol: 'protocol', thumbprint: 'thumbprint' };
var location = null;


if(typeof window !== 'undefined') {
  location = window.location;
} else {
  location = {
    protocol: 'https:'
  };
}


// constructor
// -----------

// defaults for options are handled in options_parser
function Prefetch(o) {
  this.url = o.url;
  this.ttl = o.ttl;
  this.cache = o.cache;
  this.prepare = o.prepare;
  this.transform = o.transform;
  this.transport = o.transport;
  this.thumbprint = o.thumbprint;

  this.storage = new PersistentStorage(o.cacheKey);
}

_.mixin(Prefetch.prototype, {

  _settings: function() {
    return { url: this.url, type: 'GET', dataType: 'json' };
  },

  store: function(data) {
    if (!this.cache) { return; }

    this.storage.set(keys.data, data, this.ttl);
    this.storage.set(keys.protocol, location.protocol, this.ttl);
    this.storage.set(keys.thumbprint, this.thumbprint, this.ttl);
  },

  fromCache: function() {
    var stored = {}, isExpired;

    if (!this.cache) { return null; }

    stored.data = this.storage.get(keys.data);
    stored.protocol = this.storage.get(keys.protocol);
    stored.thumbprint = this.storage.get(keys.thumbprint);

    // the stored data is considered expired if the thumbprints
    // don't match or if the protocol it was originally stored under
    // has changed
    isExpired =
      stored.thumbprint !== this.thumbprint ||
      stored.protocol !== location.protocol;

    // TODO: if expired, remove from local storage

    return stored.data && !isExpired ? stored.data : null;
  },

  fromNetwork: function(cb) {
    var that = this, settings;

    if (!cb) { return; }

    settings = this.prepare(this._settings());

    // this.transport(settings).fail(onError).done(onResponse);
    this.transport(settings).then(onResponse, onError);

    function onError() { cb(true); }
    function onResponse(resp) { cb(null, that.transform(resp)); }
  },

  clear: function() {
    this.storage.clear();
    return this;
  }
});

module.exports = Prefetch;
