<?php

namespace App\Services;

use App\Models\AccountTransaction;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class TransactionNotificationService
{
    private $oneSignalService;

    public function __construct(OneSignalService $oneSignalService)
    {
        $this->oneSignalService = $oneSignalService;
    }

    /**
     * Send notification when a transaction is created
     */
    public function sendTransactionCreatedNotification(AccountTransaction $transaction)
    {
        try {
            // Use the new method that finds user through account relationship
            return $this->oneSignalService->sendTransactionNotificationByAccount($transaction, 'transaction_completed');

        } catch (\Exception $e) {
            Log::error('Failed to send transaction created notification', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
            ]);
            
            return false;
        }
    }

    /**
     * Send notification when a transaction is approved
     */
    public function sendTransactionApprovedNotification(AccountTransaction $transaction)
    {
        try {
            return $this->oneSignalService->sendTransactionNotificationByAccount($transaction, 'transaction_approved');

        } catch (\Exception $e) {
            Log::error('Failed to send transaction approved notification', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
            ]);
            
            return false;
        }
    }

    /**
     * Send notification when a transaction is rejected
     */
    public function sendTransactionRejectedNotification(AccountTransaction $transaction)
    {
        try {
            return $this->oneSignalService->sendTransactionNotificationByAccount($transaction, 'transaction_rejected');

        } catch (\Exception $e) {
            Log::error('Failed to send transaction rejected notification', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
            ]);
            
            return false;
        }
    }

    /**
     * Get transaction created message
     */
    private function getTransactionCreatedMessage(AccountTransaction $transaction)
    {
        $amount = $transaction->transaction_amount;
        $currency = isset($transaction->from_currency_code->code) ? $transaction->from_currency_code->code : 'currency';
        
        switch ($transaction->transaction_type) {
            case 'deposit':
                return "Deposit of {$amount} {$currency} has been processed successfully.";
            
            case 'withdrawal':
                return "Withdrawal of {$amount} {$currency} has been processed successfully.";
            
            case 'transfer':
                return "Transfer of {$amount} {$currency} has been completed successfully.";
            
            case 'exchange':
                return "Currency exchange of {$amount} {$currency} has been completed successfully.";
            
            default:
                return "Transaction of {$amount} {$currency} has been processed successfully.";
        }
    }

    /**
     * Get transaction data for notification
     */
    private function getTransactionData(AccountTransaction $transaction)
    {
        return [
            'type' => 'transaction',
            'transaction_id' => $transaction->id,
            'amount' => $transaction->transaction_amount,
            'currency' => $transaction->from_currency_code->code ?? 'USD',
            'transaction_type' => $transaction->transaction_type,
            'date' => $transaction->transaction_date,
            'status' => $transaction->status,
        ];
    }
}
