<?php

namespace App\Services;

class NotificationTranslationService
{
    /**
     * Get transaction message in user's preferred language
     */
    public function getTransactionMessage($transaction, $type, $language = 'en')
    {
        switch ($type) {
            case 'transaction_completed':
                $currencyCode = isset($transaction->from_currency_code->code) ? $transaction->from_currency_code->code : $this->getCurrencyName('currency', $language);
                $formattedAmount = $this->formatNumber($transaction->transaction_amount);
                
                switch ($language) {
                    case 'ar':
                        return "تم إكمال معاملتك بمبلغ {$formattedAmount} {$currencyCode} بنجاح.";
                    case 'ckb':
                        return "مامەڵەکەت بە بڕی {$formattedAmount} {$currencyCode} بە سەرکەوتوویی تەواو بوو.";
                    default: // English
                        return "Your transaction of {$formattedAmount} {$currencyCode} has been completed successfully.";
                }
            
            case 'transaction_approved':
                switch ($language) {
                    case 'ar':
                        return "تم الموافقة على معاملتك وهي قيد المعالجة.";
                    case 'ckb':
                        return "مامەڵەکەت پشتڕاستکرایەوە و لە چارەسەرکردندایە.";
                    default: // English
                        return "Your transaction has been approved and is being processed.";
                }
            
            case 'transaction_rejected':
                switch ($language) {
                    case 'ar':
                        return "تم رفض معاملتك. يرجى الاتصال بالدعم لمزيد من المعلومات.";
                    case 'ckb':
                        return "مامەڵەکەت ڕەتکرایەوە. تکایە بۆ زانیاری زیاتر پەیوەندی بە پشتگیرییەوە بکە.";
                    default: // English
                        return "Your transaction has been rejected. Please contact support for more information.";
                }
            
            default:
                switch ($language) {
                    case 'ar':
                        return "لديك تحديث جديد في معاملتك.";
                    case 'ckb':
                        return "تۆ نوێکارییەکی نوێت هەیە لە مامەڵەکەتدا.";
                    default: // English
                        return "You have a new update on your transaction.";
                }
        }
    }

    /**
     * Get currency rate message in user's preferred language
     */
    public function getCurrencyRateMessage($currency, $newRate, $oldRate = null, $field = null, $language = 'en')
    {
        $currencyName = $currency->currency_name ?? $currency->name ?? $this->getCurrencyName('currency', $language);
        
        // Debug logging
        \Log::info('Currency rate message generation', [
            'currency' => $currencyName,
            'field' => $field,
            'oldRate' => $oldRate,
            'newRate' => $newRate,
            'language' => $language,
            'oldRateType' => gettype($oldRate),
            'newRateType' => gettype($newRate),
            'ratesEqual' => $oldRate == $newRate,
            'ratesStrictEqual' => $oldRate === $newRate,
        ]);
        
        // Format numbers with commas for better readability
        $formattedNewRate = $this->formatNumber($newRate);
        $formattedOldRate = $oldRate !== null ? $this->formatNumber($oldRate) : null;
        
        // Determine rate type based on language
        $rateType = $this->getRateType($field, $language);
        
        // Check if rates are different (with proper type conversion)
        $oldRateFloat = (float) $oldRate;
        $newRateFloat = (float) $newRate;
        
        if ($oldRate !== null && $oldRateFloat != $newRateFloat) {
            $change = $this->getRateChangeText($newRateFloat > $oldRateFloat, $language);
            
            switch ($language) {
                case 'ar':
                    $message = "سعر {$rateType} لعملة {$currencyName} من {$formattedOldRate} {$change} إلى {$formattedNewRate}";
                    break;
                case 'ckb':
                    $message = "ڕێژەی {$rateType}ی دراوی {$currencyName} لە {$formattedOldRate} {$change} بۆ {$formattedNewRate}";
                    break;
                default: // English
                    $message = "{$rateType} rate for {$currencyName} changed from {$formattedOldRate} {$change} to {$formattedNewRate}";
            }
            \Log::info('Rate change detected', ['change' => $change, 'message' => $message]);
        } else {
            switch ($language) {
                case 'ar':
                    $message = "تم تحديث سعر {$rateType} لعملة {$currencyName} إلى {$formattedNewRate}";
                    break;
                case 'ckb':
                    $message = "ڕێژەی {$rateType}ی دراوی {$currencyName} نوێکرایەوە بۆ {$formattedNewRate}";
                    break;
                default: // English
                    $message = "{$rateType} rate for {$currencyName} updated to {$formattedNewRate}";
            }
            \Log::info('No rate change detected', ['message' => $message]);
        }
        
        return $message;
    }

    /**
     * Get account update message in user's preferred language
     */
    public function getAccountUpdateMessage($account, $updateType, $language = 'en')
    {
        $accountName = $account->name ?? $this->getAccountName($language);
        
        switch ($updateType) {
            case 'balance_update':
                switch ($language) {
                    case 'ar':
                        return "تم تحديث رصيد حسابك";
                    case 'ckb':
                        return "بەڵانسی حسابەکەت نوێکرایەوە";
                    default: // English
                        return "Your account balance has been updated";
                }
            case 'account_created':
                switch ($language) {
                    case 'ar':
                        return "تم إنشاء حساب جديد لك";
                    case 'ckb':
                        return "حسابێکی نوێ بۆت دروستکرا";
                    default: // English
                        return "A new account has been created for you";
                }
            case 'account_updated':
                switch ($language) {
                    case 'ar':
                        return "تم تحديث معلومات حسابك";
                    case 'ckb':
                        return "زانیارییەکانی حسابەکەت نوێکرانەوە";
                    default: // English
                        return "Your account information has been updated";
                }
            default:
                switch ($language) {
                    case 'ar':
                        return "تم تحديث حسابك";
                    case 'ckb':
                        return "حسابەکەت نوێکرایەوە";
                    default: // English
                        return "Your account has been updated";
                }
        }
    }

    /**
     * Get test notification message in user's preferred language
     */
    public function getTestMessage($language = 'en')
    {
        $timestamp = now()->format('Y-m-d H:i:s');
        
        switch ($language) {
            case 'ar':
                return "اختبار الإشعارات في كاش إن آر إكس - {$timestamp}";
            case 'ckb':
                return "تەستکردنی ئاگادارکردنەوە لە الامان - {$timestamp}";
            default: // English
                return "Test notification from Double Hair - {$timestamp}";
        }
    }

    /**
     * Get welcome message in user's preferred language
     */
    public function getWelcomeMessage($userName = null, $language = 'en')
    {
        switch ($language) {
            case 'ar':
                if ($userName) {
                    return "مرحباً بك في كاش إن آر إكس، {$userName}";
                }
                return "مرحباً بك في كاش إن آر إكس";
            case 'ckb':
                if ($userName) {
                    return "بەخێربێیت بۆ الامان، {$userName}";
                }
                return "بەخێربێیت بۆ الامان";
            default: // English
                if ($userName) {
                    return "Welcome to Double Hair, {$userName}";
                }
                return "Welcome to Double Hair";
        }
    }

    /**
     * Get payment received message in user's preferred language
     */
    public function getPaymentReceivedMessage($amount, $currency = null, $language = 'en')
    {
        $formattedAmount = $this->formatNumber($amount);
        $currencyText = $currency ?? $this->getCurrencyName('dinar', $language);
        
        switch ($language) {
            case 'ar':
                return "تم استلام مبلغ {$formattedAmount} {$currencyText} بنجاح";
            case 'ckb':
                return "بڕی {$formattedAmount} {$currencyText} وەرگیرا بە سەرکەوتوویی";
            default: // English
                return "Payment of {$formattedAmount} {$currencyText} received successfully";
        }
    }

    /**
     * Get low balance warning message in user's preferred language
     */
    public function getLowBalanceWarningMessage($balance, $currency = null, $language = 'en')
    {
        $formattedBalance = $this->formatNumber($balance);
        $currencyText = $currency ?? $this->getCurrencyName('dinar', $language);
        
        switch ($language) {
            case 'ar':
                return "تحذير: رصيد حسابك منخفض ({$formattedBalance} {$currencyText}). يرجى إضافة أموال.";
            case 'ckb':
                return "ئاگاداری: بەڵانسی حسابەکەت کەمە ({$formattedBalance} {$currencyText}). تکایە پارە زیاد بکە.";
            default: // English
                return "Warning: Your account balance is low ({$formattedBalance} {$currencyText}). Please add funds.";
        }
    }

    /**
     * Get security alert message in user's preferred language
     */
    public function getSecurityAlertMessage($language = 'en')
    {
        switch ($language) {
            case 'ar':
                return "تنبيه أمني: تم اكتشاف نشاط مشبوه في حسابك. يرجى تغيير كلمة المرور.";
            case 'ckb':
                return "ئاگاداری سەلامەتی: چالاکییەکی گوماناوی لە حسابەکەتدا دۆزرایەوە. تکایە پاسۆردەکەت بگۆڕە.";
            default: // English
                return "Security Alert: Suspicious activity detected on your account. Please change your password.";
        }
    }

    /**
     * Format number with commas for better readability
     */
    private function formatNumber($number)
    {
        // Convert to float first to handle any string numbers
        $number = (float) $number;
        
        // Format with commas for thousands separator
        return number_format($number, 0, '.', ',');
    }

    /**
     * Get rate type text based on field and language
     */
    private function getRateType($field, $language)
    {
        switch ($language) {
            case 'ar':
                if ($field === 'buy_rate') {
                    return 'الشراء';
                } elseif ($field === 'sell_rate') {
                    return 'البيع';
                }
                return 'السعر';
            case 'ckb':
                if ($field === 'buy_rate') {
                    return 'کڕین';
                } elseif ($field === 'sell_rate') {
                    return 'فرۆشتن';
                }
                return 'ڕێژە';
            default: // English
                if ($field === 'buy_rate') {
                    return 'Buy';
                } elseif ($field === 'sell_rate') {
                    return 'Sell';
                }
                return 'Rate';
        }
    }

    /**
     * Get rate change text based on increase/decrease and language
     */
    private function getRateChangeText($isIncrease, $language)
    {
        switch ($language) {
            case 'ar':
                return $isIncrease ? 'ارتفع' : 'انخفض';
            case 'ckb':
                return $isIncrease ? 'زیادبووە' : 'کەمبووەتەوە';
            default: // English
                return $isIncrease ? 'increased' : 'decreased';
        }
    }

    /**
     * Get currency name based on language
     */
    private function getCurrencyName($type, $language)
    {
        switch ($language) {
            case 'ar':
                if ($type === 'dinar') {
                    return 'دينار';
                }
                return 'عملة';
            case 'ckb':
                if ($type === 'dinar') {
                    return 'دینار';
                }
                return 'دراو';
            default: // English
                if ($type === 'dinar') {
                    return 'Dinar';
                }
                return 'Currency';
        }
    }

    /**
     * Get account name based on language
     */
    private function getAccountName($language)
    {
        switch ($language) {
            case 'ar':
                return 'حسابك';
            case 'ckb':
                return 'حسابەکەت';
            default: // English
                return 'your account';
        }
    }
}




