<?php

namespace App\Models;

use App\Models\Scopes\Searchable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class ProductBatch extends Model
{
    use HasFactory;
    use Searchable;
    use LogsActivity;

    protected $fillable = [
        'product_id',
        'batch_number',
        'purchase_price',
        'expired_date',
        'quantity',
        'sold_quantity',
        'returned_quantity',
        'purchase_date',
    ];

    protected $searchableFields = ['*'];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'expired_date' => 'date',
        'purchase_date' => 'date',
        'quantity' => 'integer',
        'sold_quantity' => 'integer',
        'returned_quantity' => 'integer',
    ];

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function invoiceItems()
    {
        return $this->hasMany(InvoiceItem::class);
    }

    /**
     * Get available quantity for this batch
     */
    public function getAvailableQuantityAttribute()
    {
        return $this->quantity - $this->sold_quantity + $this->returned_quantity;
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['created_at', 'updated_at'])
            ->logOnlyDirty()
            ->setDescriptionForEvent(
                fn(string $eventName) => "ProductBatch has been {$eventName}"
            );
    }
}
