<?php

namespace App\Models;

use App\Models\Scopes\Searchable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Product extends Model
{
    use HasFactory;
    use Searchable;
    use LogsActivity;

    protected $fillable = [
        'name',
        'sale_price',
        'purchase_price',
        'supplier_id',
        'expired_date',
        'stock_quantity',
        'description',
    ];

    protected $searchableFields = ['*'];

    protected $casts = [
        'sale_price' => 'decimal:2',
        'purchase_price' => 'decimal:2',
        'expired_date' => 'date',
        'stock_quantity' => 'integer',
    ];

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function batches()
    {
        return $this->hasMany(ProductBatch::class);
    }

    public function invoiceItems()
    {
        return $this->hasMany(InvoiceItem::class);
    }

    /**
     * Calculate total available stock from all batches
     */
    public function getAvailableStockAttribute()
    {
        return $this->batches()->sum('quantity') - 
               $this->batches()->sum('sold_quantity') + 
               $this->batches()->sum('returned_quantity');
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['created_at', 'updated_at'])
            ->logOnlyDirty()
            ->setDescriptionForEvent(
                fn(string $eventName) => "Product has been {$eventName}"
            );
    }
}
