<?php

namespace App\Livewire;

use App\Models\Account;
use App\Models\Transaction;
use App\Models\AccountTransaction;
use App\Models\Customer;
use App\Models\User;
use App\Models\Currency;
use App\Models\CashierShift;
use App\Models\Cashier;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class Search extends Component
{
    public $query = '';
    public $searchResults = [];
    public $totalResults = 0;
    public $searching = false;
    public $recentSearches = [];
    public $selectedCategory = 'all';
    public $showResults = false;

    protected $queryString = ['query'];

    public function mount()
    {
        $this->loadRecentSearches();
    }

    public function updatedQuery()
    {
        $this->showResults = false;
        
        if (strlen($this->query) >= 2) {
            $this->searching = true;
            $this->performSearch();
            $this->searching = false;
            $this->showResults = true;
        } else {
            $this->searchResults = [];
            $this->totalResults = 0;
        }
    }

    public function performSearch()
    {
        $query = $this->query;
        $results = [];

        // Search Transactions
        $transactions = Transaction::with(['from_currency_code', 'to_currency_code', 'cashierShift'])
            ->where(function($q) use ($query) {
                $q->where('id', 'like', "%{$query}%")
                  ->orWhere('transaction_amount', 'like', "%{$query}%")
                  ->orWhere('transaction_type', 'like', "%{$query}%")
                  ->orWhere('amount_given', 'like', "%{$query}%")
                  ->orWhere('amount_received', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($transaction) {
                return [
                    'type' => 'transaction',
                    'id' => $transaction->id,
                    'title' => "جووڵە #{$transaction->id}",
                    'subtitle' => $transaction->transaction_type . ' - ' . number_format($transaction->transaction_amount),
                    'description' => ($transaction->from_currency_code->currency_name ?? '') . ' بۆ ' . ($transaction->to_currency_code->currency_name ?? ''),
                    'date' => $transaction->created_at->format('Y-m-d'),
                    'url' => route('transaction_invoice', $transaction->id),
                    'icon' => 'ti-exchange',
                    'badge' => $transaction->transaction_type,
                    'badge_color' => 'primary'
                ];
            });

        // Search Account Transactions
        $accountTransactions = AccountTransaction::with(['from_account', 'to_account', 'user'])
            ->where(function($q) use ($query) {
                $q->where('id', 'like', "%{$query}%")
                  ->orWhere('transaction_amount', 'like', "%{$query}%")
                  ->orWhere('person_name', 'like', "%{$query}%")
                  ->orWhere('sender_name', 'like', "%{$query}%")
                  ->orWhere('receiver_name', 'like', "%{$query}%")
                  ->orWhere('sender_phone', 'like', "%{$query}%")
                  ->orWhere('receiver_phone', 'like', "%{$query}%")
                  ->orWhere('note', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($transaction) {
                $statusColor = $transaction->status == 'approved' ? 'success' : ($transaction->status == 'rejected' ? 'danger' : 'warning');
                
                // Determine the URL based on the from_account
                $url = '#';
                if ($transaction->from_account) {
                    $url = in_array($transaction->from_account->customer?->customer_name, ['Cashiers', 'Bank (Cashboxes)']) 
                        ? route('show-cashbox', $transaction->from_account->id) 
                        : route('accounts.show', $transaction->from_account->id);
                }
                
                return [
                    'type' => 'account_transaction',
                    'id' => $transaction->id,
                    'title' => "جووڵەی هەژمار #{$transaction->id}",
                    'subtitle' => ($transaction->person_name ?? $transaction->sender_name ?? '-') . ' - ' . number_format($transaction->transaction_amount),
                    'description' => $transaction->transaction_type . ' - ' . ($transaction->note ?? ''),
                    'date' => $transaction->transaction_date?->format('Y-m-d') ?? $transaction->created_at->format('Y-m-d'),
                    'url' => $url,
                    'icon' => 'ti-wallet',
                    'badge' => $transaction->status ?? 'pending',
                    'badge_color' => $statusColor
                ];
            });

        // Search Accounts
        $accounts = Account::with(['customer', 'currency'])
            ->where(function($q) use ($query) {
                $q->where('account_title', 'like', "%{$query}%")
                  ->orWhere('account_number', 'like', "%{$query}%")
                  ->orWhere('balance', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($account) {
                $url = in_array($account->customer?->customer_name, ['Cashiers', 'Bank (Cashboxes)']) 
                    ? route('show-cashbox', $account->id) 
                    : route('accounts.show', $account->id);
                
                return [
                    'type' => 'account',
                    'id' => $account->id,
                    'title' => $account->account_title,
                    'subtitle' => 'باڵانس: ' . number_format($account->balance) . ' ' . ($account->currency->currency_name ?? ''),
                    'description' => 'کڕیار: ' . ($account->customer?->customer_name ?? '-'),
                    'date' => $account->created_at->format('Y-m-d'),
                    'url' => $url,
                    'icon' => 'ti-credit-card',
                    'badge' => $account->account_type ?? 'account',
                    'badge_color' => 'info'
                ];
            });

        // Search Customers
        $customers = Customer::with('user')
            ->where(function($q) use ($query) {
                $q->where('customer_name', 'like', "%{$query}%")
                  ->orWhere('phone_number', 'like', "%{$query}%")
                  ->orWhere('second_phone', 'like', "%{$query}%")
                  ->orWhere('address', 'like', "%{$query}%")
                  ->orWhere('note', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($customer) {
                return [
                    'type' => 'customer',
                    'id' => $customer->id,
                    'title' => $customer->customer_name,
                    'subtitle' => $customer->phone_number ?? '-',
                    'description' => $customer->address ?? $customer->note ?? '-',
                    'date' => $customer->created_at->format('Y-m-d'),
                    'url' => route('show_customer', $customer->id),
                    'icon' => 'ti-user',
                    'badge' => 'کڕیار',
                    'badge_color' => 'success'
                ];
            });

        // Search Users
        $users = User::where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($user) {
                return [
                    'type' => 'user',
                    'id' => $user->id,
                    'title' => $user->name,
                    'subtitle' => $user->email,
                    'description' => 'ڕۆڵەکان: ' . $user->roles->pluck('name')->join(', '),
                    'date' => $user->created_at->format('Y-m-d'),
                    'url' => route('users.show', $user->id),
                    'icon' => 'ti-user-circle',
                    'badge' => 'بەکارهێنەر',
                    'badge_color' => 'primary'
                ];
            });

        // Search Currencies
        $currencies = Currency::where(function($q) use ($query) {
                $q->where('currency_name', 'like', "%{$query}%")
                  ->orWhere('code', 'like', "%{$query}%")
                  ->orWhere('type', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($currency) {
                return [
                    'type' => 'currency',
                    'id' => $currency->id,
                    'title' => $currency->currency_name,
                    'subtitle' => $currency->code . ' - ' . number_format($currency->rate, 2),
                    'description' => 'جۆر: ' . ($currency->type ?? '-'),
                    'date' => $currency->created_at->format('Y-m-d'),
                    'url' => route('currencies.show', $currency->id),
                    'icon' => 'ti-currency-dollar',
                    'badge' => 'دراو',
                    'badge_color' => 'warning'
                ];
            });

        // Search Cashier Shifts
        $cashierShifts = CashierShift::with(['cashier', 'supervisor'])
            ->where(function($q) use ($query) {
                $q->where('id', 'like', "%{$query}%")
                  ->orWhere('starting_amount', 'like', "%{$query}%")
                  ->orWhere('ending_amount', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($shift) {
                return [
                    'type' => 'cashier_shift',
                    'id' => $shift->id,
                    'title' => "شیفتی کاشیر #{$shift->id}",
                    'subtitle' => ($shift->cashier?->cashier_name ?? 'نەدۆزرایەوە') . ' - ' . $shift->start_time,
                    'description' => 'بڕی سەرەتا: ' . number_format($shift->starting_amount ?? 0),
                    'date' => $shift->created_at->format('Y-m-d'),
                    'url' => '#',
                    'icon' => 'ti-briefcase',
                    'badge' => 'شیفت',
                    'badge_color' => 'secondary'
                ];
            });

        // Combine all results
        $this->searchResults = [
            'transactions' => $transactions,
            'account_transactions' => $accountTransactions,
            'accounts' => $accounts,
            'customers' => $customers,
            'users' => $users,
            'currencies' => $currencies,
            'cashier_shifts' => $cashierShifts,
        ];

        // Calculate total results
        $this->totalResults = 
            $transactions->count() +
            $accountTransactions->count() +
            $accounts->count() +
            $customers->count() +
            $users->count() +
            $currencies->count() +
            $cashierShifts->count();

        // Save to recent searches if we have results
        if ($this->totalResults > 0) {
            $this->saveRecentSearch($this->query);
        }
    }

    public function saveRecentSearch($query)
    {
        $recentSearches = session()->get('recent_searches', []);
        
        // Remove if already exists
        $recentSearches = array_filter($recentSearches, function($item) use ($query) {
            return $item !== $query;
        });
        
        // Add to beginning
        array_unshift($recentSearches, $query);
        
        // Keep only last 5
        $recentSearches = array_slice($recentSearches, 0, 5);
        
        session()->put('recent_searches', $recentSearches);
        $this->recentSearches = $recentSearches;
    }

    public function loadRecentSearches()
    {
        $this->recentSearches = session()->get('recent_searches', []);
    }

    public function searchFromRecent($query)
    {
        $this->query = $query;
        $this->performSearch();
        $this->showResults = true;
    }

    public function clearRecentSearches()
    {
        session()->forget('recent_searches');
        $this->recentSearches = [];
    }

    public function render()
    {
        return view('livewire.search');
    }
}
