<?php

namespace App\Livewire;

use App\Models\Product;
use App\Models\Supplier;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use App\Traits\HasUtf8Encoding;
use Carbon\Carbon;

class ProductReport extends Component
{
    use LivewireAlert, WithPagination, HasUtf8Encoding;

    protected $paginationTheme = 'bootstrap';

    // Filters
    public $search = '';
    public $supplier_id = '';
    public $stockFilter = 'all'; // all, in_stock, low_stock, out_of_stock
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';

    public function mount()
    {
        // Initialize filters
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedSupplierId()
    {
        $this->resetPage();
    }
    
    public function updatedSupplier_id()
    {
        $this->resetPage();
    }

    public function updatedStockFilter()
    {
        $this->resetPage();
    }

    public function sortByField($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->supplier_id = '';
        $this->stockFilter = 'all';
        $this->sortBy = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }

    public function getAnalysisData()
    {
        $query = $this->getBaseQuery();

        $allProducts = Product::with(['supplier', 'invoiceItems'])->get();
        $filteredProducts = $query->with(['supplier', 'invoiceItems'])->get();

        $totalStockValue = $allProducts->sum(function($product) {
            return ($product->stock_quantity ?? 0) * ($product->purchase_price ?? 0);
        });

        $filteredStockValue = $filteredProducts->sum(function($product) {
            return ($product->stock_quantity ?? 0) * ($product->purchase_price ?? 0);
        });

        $totalSalesValue = $allProducts->sum(function($product) {
            return $product->invoiceItems->sum('total_price');
        });

        $filteredSalesValue = $filteredProducts->sum(function($product) {
            return $product->invoiceItems->sum('total_price');
        });

        return [
            'total_products' => Product::count(),
            'filtered_products' => $query->count(),
            'total_stock_value' => $totalStockValue,
            'filtered_stock_value' => $filteredStockValue,
            'total_sales_value' => $totalSalesValue,
            'filtered_sales_value' => $filteredSalesValue,
            'low_stock_count' => Product::where('stock_quantity', '<', 10)->count(),
            'out_of_stock_count' => Product::where('stock_quantity', '<=', 0)->count(),
            'total_sold_items' => $allProducts->sum(function($product) {
                return $product->invoiceItems->sum('quantity');
            }),
        ];
    }

    private function getBaseQuery()
    {
        $query = Product::query();

        // Search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%');
            });
        }

        // Supplier filter
        if ($this->supplier_id) {
            $query->where('supplier_id', $this->supplier_id);
        }

        // Stock filter
        if ($this->stockFilter === 'in_stock') {
            $query->where('stock_quantity', '>', 0);
        } elseif ($this->stockFilter === 'low_stock') {
            $query->where('stock_quantity', '>', 0)->where('stock_quantity', '<', 10);
        } elseif ($this->stockFilter === 'out_of_stock') {
            $query->where('stock_quantity', '<=', 0);
        }

        return $query;
    }

    public function render()
    {
        $query = $this->getBaseQuery();

        // Add relationships
        $query->with(['supplier'])->withCount(['invoiceItems', 'batches']);

        // Sorting
        $query->orderBy($this->sortBy, $this->sortDirection);

        $products = $query->paginate(25);

        $analysis = $this->getAnalysisData();
        $suppliers = Supplier::orderBy('name')->get();

        return view('livewire.product-report', [
            'products' => $products,
            'analysis' => $analysis,
            'suppliers' => $suppliers,
        ]);
    }
}

