<?php

namespace App\Livewire;

use App\Models\Patient;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use App\Traits\HasUtf8Encoding;
use Carbon\Carbon;

class PatientReport extends Component
{
    use LivewireAlert, WithPagination, HasUtf8Encoding;

    protected $paginationTheme = 'bootstrap';

    // Filters
    public $search = '';
    public $startDate = '';
    public $endDate = '';
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';

    public function mount()
    {
        // Set default date range to last 30 days
        $this->endDate = now()->format('Y-m-d');
        $this->startDate = now()->subDays(30)->format('Y-m-d');
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedStartDate()
    {
        $this->resetPage();
    }

    public function updatedEndDate()
    {
        $this->resetPage();
    }

    public function sortByField($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->startDate = now()->subDays(30)->format('Y-m-d');
        $this->endDate = now()->format('Y-m-d');
        $this->sortBy = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }

    public function getAnalysisData()
    {
        $query = $this->getBaseQuery();

        return [
            'total_patients' => Patient::count(),
            'filtered_patients' => $query->count(),
            'total_invoices' => Patient::withCount('invoices')->get()->sum('invoices_count'),
            'filtered_invoices' => $query->withCount('invoices')->get()->sum('invoices_count'),
            'total_histories' => Patient::withCount('histories')->get()->sum('histories_count'),
            'filtered_histories' => $query->withCount('histories')->get()->sum('histories_count'),
            'new_patients_today' => Patient::whereDate('created_at', today())->count(),
            'new_patients_this_month' => Patient::whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)->count(),
        ];
    }

    private function getBaseQuery()
    {
        $query = Patient::query();

        // Search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('code', 'like', '%' . $this->search . '%')
                  ->orWhere('phone', 'like', '%' . $this->search . '%')
                  ->orWhere('address', 'like', '%' . $this->search . '%');
            });
        }

        // Date filter
        if ($this->startDate) {
            $query->whereDate('created_at', '>=', $this->startDate);
        }

        if ($this->endDate) {
            $query->whereDate('created_at', '<=', $this->endDate);
        }

        return $query;
    }

    public function render()
    {
        $query = $this->getBaseQuery();

        // Add relationships for counting
        $query->withCount(['invoices', 'histories']);

        // Sorting
        $query->orderBy($this->sortBy, $this->sortDirection);

        $patients = $query->paginate(25);

        $analysis = $this->getAnalysisData();

        return view('livewire.patient-report', [
            'patients' => $patients,
            'analysis' => $analysis,
        ]);
    }
}

