<?php

namespace App\Livewire;

use App\Models\Patient;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use App\Traits\HasUtf8Encoding;

class PatientManagement extends Component
{
    use LivewireAlert, WithPagination, HasUtf8Encoding;

    protected $paginationTheme = 'bootstrap';
    
    public $searchQuery = '';
    public $showCreateModal = false;
    public $showEditModal = false;
    
    // Create form properties
    public $code, $name, $phone, $address;
    
    // Edit form properties
    public $editPatientId, $edit_code, $edit_name, $edit_phone, $edit_address;

    public function mount()
    {
        // Check for session errors and display them
        if (session()->has('errors') && session('errors')->has('error')) {
            $this->alert('error', session('errors')->first('error'));
        }
        
        if (session('success')) {
            $this->alert('success', session('success'));
        }
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->showCreateModal = true;
    }

    public function closeCreateModal()
    {
        $this->resetForm();
        $this->showCreateModal = false;
    }

    public function openEditModal()
    {
        $this->showEditModal = true;
    }

    public function closeEditModal()
    {
        $this->resetEditForm();
        $this->showEditModal = false;
    }

    public function savePatient()
    {
        // Trim inputs before validation
        $this->code = trim($this->code ?? '');
        $this->name = trim($this->name ?? '');
        $this->phone = trim($this->phone ?? '');
        $this->address = $this->address ? trim($this->address) : null;

        $validated = $this->validate([
            'code' => 'required|string|max:255|unique:patients,code',
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:255',
            'address' => 'nullable|string',
        ], [
            'code.required' => 'Patient code is required.',
            'code.unique' => 'This patient code already exists. Please use a different code.',
            'code.max' => 'Patient code cannot exceed 255 characters.',
            'name.required' => 'Patient name is required.',
            'name.max' => 'Patient name cannot exceed 255 characters.',
            'phone.required' => 'Phone number is required.',
            'phone.max' => 'Phone number cannot exceed 255 characters.',
        ]);

        try {
            Patient::create([
                'code' => $validated['code'],
                'name' => $validated['name'],
                'phone' => $validated['phone'],
                'address' => $validated['address'] ?? null,
            ]);

            $this->alert('success', 'Patient created successfully!');
            $this->closeCreateModal();
            $this->resetPage(); // Reset pagination
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to create patient: ' . $e->getMessage());
        }
    }

    public function showEditPatientModal($patientId)
    {
        $patient = Patient::findOrFail($patientId);

        $this->editPatientId = $patient->id;
        $this->edit_code = $patient->code;
        $this->edit_name = $patient->name;
        $this->edit_phone = $patient->phone;
        $this->edit_address = $patient->address;
        
        $this->openEditModal();
    }

    public function updatePatient()
    {
        $this->validate([
            'edit_code' => 'required|string|max:255|unique:patients,code,' . $this->editPatientId,
            'edit_name' => 'required|string|max:255',
            'edit_phone' => 'required|string|max:255',
            'edit_address' => 'nullable|string',
        ], [
            'edit_code.required' => 'Patient code is required.',
            'edit_code.unique' => 'This patient code already exists. Please use a different code.',
            'edit_code.max' => 'Patient code cannot exceed 255 characters.',
            'edit_name.required' => 'Patient name is required.',
            'edit_name.max' => 'Patient name cannot exceed 255 characters.',
            'edit_phone.required' => 'Phone number is required.',
            'edit_phone.max' => 'Phone number cannot exceed 255 characters.',
        ]);

        try {
            $patient = Patient::findOrFail($this->editPatientId);
            $patient->update([
                'code' => trim($this->edit_code),
                'name' => trim($this->edit_name),
                'phone' => trim($this->edit_phone),
                'address' => $this->edit_address ? trim($this->edit_address) : null,
            ]);

            $this->alert('success', 'Patient updated successfully!');
            $this->closeEditModal();
            $this->resetPage(); // Reset pagination
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to update patient: ' . $e->getMessage());
        }
    }

    public function deletePatient($patientId)
    {
        try {
            $patient = Patient::findOrFail($patientId);
            
            // Check for related data
            if ($patient->invoices()->count() > 0) {
                $this->alert('error', 'Cannot delete patient. This patient has associated invoices.');
                return;
            }

            if ($patient->histories()->count() > 0) {
                $this->alert('error', 'Cannot delete patient. This patient has associated histories.');
                return;
            }

            $patient->delete();
            $this->alert('success', 'Patient deleted successfully!');
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                $this->alert('error', 'Cannot delete patient. This patient has related data that prevents deletion.');
            } else {
                $this->alert('error', 'An error occurred while deleting the patient. Please try again.');
            }
        } catch (\Exception $e) {
            $this->alert('error', 'An unexpected error occurred. Please try again.');
        }
    }

    private function resetForm()
    {
        $this->name = '';
        $this->phone = '';
        $this->address = '';
        $this->clinic_id = null;
    }

    private function resetEditForm()
    {
        $this->editPatientId = null;
        $this->edit_name = '';
        $this->edit_phone = '';
        $this->edit_address = '';
        $this->edit_clinic_id = null;
    }

    public function render()
    {
        $query = Patient::query();

        if ($this->searchQuery) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->searchQuery . '%')
                  ->orWhere('code', 'like', '%' . $this->searchQuery . '%')
                  ->orWhere('phone', 'like', '%' . $this->searchQuery . '%');
            });
        }

        return view('livewire.patient-management', [
            'patients' => $query->paginate(50),
        ]);
    }
}
