<?php

namespace App\Livewire;

use App\Models\Patient;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Product;
use App\Models\ProductBatch;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use App\Traits\HasUtf8Encoding;

class PatientInvoiceWizard extends Component
{
    use LivewireAlert, HasUtf8Encoding;

    // Modal states
    public $showServiceModal = false;
    public $showProductModal = false;

    // Step 1: Patient form properties
    public $name, $phone, $address;
    public $searchQuery = '';
    public $searchResults = [];
    public $showSearchResults = false;
    public $selectedPatientId = null;
    public $isEditingPatient = false;

    // Step 2: Invoice form properties
    public $invoice_date, $notes; // clinic_id removed from UI
    
    // Step 2: Search properties for services
    public $serviceSearchQuery = '';
    public $serviceSearchResults = [];
    public $showServiceSearchResults = false;
    
    // Step 2: Search properties for products (drugs)
    public $productSearchQuery = '';
    public $productSearchResults = [];
    public $showProductSearchResults = false;
    
    // Step 2: Selected items (using list format)
    public $selectedServices = []; // Array of [id => ['subcategory_id', 'category_id', 'name', 'cost']]
    public $selectedProducts = []; // Array of [id => ['product_id', 'name', 'quantity', 'batch_id', 'sale_price']]
    public $selectedProductBatches = []; // Array of [product_id => batch_id]
    
    // Available data
    public $categories = [];
    public $subcategories = [];
    public $products = [];
    public $productBatches = [];

    // Review data (computed properties)

    public function mount()
    {
        $this->categories = Category::with('subcategories')->get();
        $this->products = Product::with('supplier', 'batches')->get();
        $this->invoice_date = now()->format('Y-m-d'); // Keep for saving, but don't show in UI
    }

    public function nextStep()
    {
        if ($this->currentStep == 1) {
            $this->validateStep1();
        } elseif ($this->currentStep == 2) {
            if (!$this->validateStep2()) {
                return;
            }
            $this->prepareReview();
        }
        
        if ($this->currentStep < $this->totalSteps) {
            $this->currentStep++;
        }
    }

    public function previousStep()
    {
        if ($this->currentStep > 1) {
            $this->currentStep--;
        }
    }

    public function goToStep($step)
    {
        // Only allow going to steps that are already completed or the current/next step
        if ($step >= 1 && $step <= $this->totalSteps) {
            // Can always go to step 1
            if ($step == 1) {
                $this->currentStep = $step;
            }
            // Can go to step 2 only if we're on step 1 or already past it
            elseif ($step == 2) {
                if ($this->currentStep == 1) {
                    // Validate step 1 before going to step 2
                    try {
                        $this->validateStep1();
                        $this->currentStep = $step;
                    } catch (\Illuminate\Validation\ValidationException $e) {
                        // Stay on step 1 if validation fails
                        return;
                    }
                } elseif ($this->currentStep >= 2) {
                    // Already completed step 1, can go back to step 2
                    $this->currentStep = $step;
                }
            }
            // Can go to step 3 only if we're on step 2 or already past it
            elseif ($step == 3) {
                if ($this->currentStep == 2) {
                    // Validate step 2 before going to step 3
                    if ($this->validateStep2()) {
                        $this->prepareReview();
                        $this->currentStep = $step;
                    }
                    // Stay on step 2 if validation fails
                } elseif ($this->currentStep >= 3) {
                    // Already completed step 2, can go back to step 3
                    $this->currentStep = $step;
                }
            }
        }
    }

    private function validateStep1()
    {
        // If editing existing patient, validate that patient exists
        if ($this->isEditingPatient && $this->selectedPatientId) {
            $this->validate([
                'selectedPatientId' => 'required|exists:patients,id',
                'name' => 'required|string|max:255',
                'phone' => 'nullable|string|max:255',
                'address' => 'nullable|string',
            ], [
                'selectedPatientId.required' => 'Please select a patient.',
                'selectedPatientId.exists' => 'Selected patient does not exist.',
                'name.required' => 'Patient name is required.',
            ]);
        } else {
            // Creating new patient - code is auto-generated
            $this->validate([
                'name' => 'required|string|max:255',
                'phone' => 'nullable|string|max:255',
                'address' => 'nullable|string',
            ], [
                'name.required' => 'Patient name is required.',
            ]);
        }
    }

    private function validateStep2()
    {
        $hasServices = !empty($this->selectedServices);
        $hasDrugs = !empty($this->selectedProducts);
        
        // Invoice can have only services, only drugs, or both
        if (!$hasServices && !$hasDrugs) {
            $this->addError('items', 'Please add at least one service or product (drug) for the invoice.');
            return false;
        }
        
        // Validate product quantities
        foreach ($this->selectedProducts as $index => $product) {
            if ($product['quantity'] <= 0 || $product['quantity'] > $product['stock_quantity']) {
                $this->addError('items', 'Invalid quantity for product: ' . $product['name']);
                return false;
            }
        }
        
        return true;
    }


    public function updatedName($value)
    {
        if ($this->isEditingPatient && $this->selectedPatientId) {
            // Don't search if we're editing an existing patient
            return;
        }
        
        $this->searchQuery = $value;
        if (strlen($value) >= 1) {
            $this->searchPatients('name', $value);
        } else {
            // Just hide search results, don't clear the input
            $this->searchResults = [];
            $this->showSearchResults = false;
        }
    }

    public function searchPatients($field = null, $value = null)
    {
        $searchValue = $value ?? $this->searchQuery;
        
        if (empty($searchValue)) {
            $this->searchResults = [];
            $this->showSearchResults = false;
            return;
        }

        $query = Patient::query();

        // Search by code or name
        $query->where(function($q) use ($searchValue) {
            $q->where('code', 'like', '%' . $searchValue . '%')
              ->orWhere('name', 'like', '%' . $searchValue . '%');
        });

        $this->searchResults = $query->limit(10)->get();
        $this->showSearchResults = count($this->searchResults) > 0;
    }

    public function selectPatient($patientId)
    {
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->selectedPatientId = $patient->id;
            $this->name = $patient->name;
            $this->phone = $patient->phone;
            $this->address = $patient->address;
            $this->isEditingPatient = true;
            $this->searchResults = [];
            $this->showSearchResults = false;
            $this->searchQuery = '';
        }
    }

    public function resetPatientFields()
    {
        // Only reset if not editing a patient and fields are truly empty
        if (!$this->isEditingPatient && empty($this->name)) {
            $this->phone = '';
            $this->address = '';
            $this->selectedPatientId = null;
        }
    }

    public function clearPatientSelection()
    {
        $this->selectedPatientId = null;
        $this->isEditingPatient = false;
        $this->name = '';
        $this->phone = '';
        $this->address = '';
        $this->searchQuery = '';
        $this->searchResults = [];
        $this->showSearchResults = false;
    }

    // Service search and management
    public function updatedServiceSearchQuery($value)
    {
        if (strlen($value) >= 1) {
            $this->searchServices($value);
        } else {
            $this->serviceSearchResults = [];
            $this->showServiceSearchResults = false;
        }
    }

    public function searchServices($query = null)
    {
        $searchValue = $query ?? $this->serviceSearchQuery;
        
        if (empty($searchValue)) {
            $this->serviceSearchResults = [];
            $this->showServiceSearchResults = false;
            return;
        }

        $serviceCategories = Category::whereIn('type', ['service', 'other'])
            ->with(['subcategories' => function($q) use ($searchValue) {
                $q->where('name', 'like', '%' . $searchValue . '%');
            }])
            ->get();

        $results = [];
        foreach ($serviceCategories as $category) {
            foreach ($category->subcategories as $subcategory) {
                $results[] = [
                    'id' => $subcategory->id,
                    'category_id' => $category->id,
                    'category_name' => $category->name,
                    'name' => $subcategory->name,
                    'cost' => $subcategory->cost,
                ];
            }
        }

        $this->serviceSearchResults = $results;
        $this->showServiceSearchResults = count($results) > 0;
    }

    public function addService($subcategoryId, $categoryId, $name, $cost)
    {
        // Check if already added
        $exists = false;
        foreach ($this->selectedServices as $service) {
            if ($service['subcategory_id'] == $subcategoryId) {
                $exists = true;
                break;
            }
        }

        if (!$exists) {
            $this->selectedServices[] = [
                'subcategory_id' => $subcategoryId,
                'category_id' => $categoryId,
                'name' => $name,
                'cost' => $cost,
            ];
        }

        $this->serviceSearchQuery = '';
        $this->serviceSearchResults = [];
        $this->showServiceSearchResults = false;
        $this->closeServiceModal();
    }

    public function openServiceModal()
    {
        $this->showServiceModal = true;
        $this->serviceSearchQuery = '';
        $this->serviceSearchResults = [];
        $this->showServiceSearchResults = false;
    }

    public function closeServiceModal()
    {
        $this->showServiceModal = false;
        $this->serviceSearchQuery = '';
        $this->serviceSearchResults = [];
        $this->showServiceSearchResults = false;
    }

    public function openProductModal()
    {
        $this->showProductModal = true;
        $this->productSearchQuery = '';
        $this->productSearchResults = [];
        $this->showProductSearchResults = false;
    }

    public function closeProductModal()
    {
        $this->showProductModal = false;
        $this->productSearchQuery = '';
        $this->productSearchResults = [];
        $this->showProductSearchResults = false;
    }

    public function removeService($index)
    {
        if (isset($this->selectedServices[$index])) {
            unset($this->selectedServices[$index]);
            $this->selectedServices = array_values($this->selectedServices);
        }
    }

    // Product (drug) search and management
    public function updatedProductSearchQuery($value)
    {
        if (strlen($value) >= 1) {
            $this->searchProducts($value);
        } else {
            $this->productSearchResults = [];
            $this->showProductSearchResults = false;
        }
    }

    public function searchProducts($query = null)
    {
        $searchValue = $query ?? $this->productSearchQuery;
        
        if (empty($searchValue)) {
            $this->productSearchResults = [];
            $this->showProductSearchResults = false;
            return;
        }

        $products = Product::where('name', 'like', '%' . $searchValue . '%')
            ->where('stock_quantity', '>', 0)
            ->with(['supplier', 'batches' => function($q) {
                $q->whereRaw('quantity - sold_quantity + returned_quantity > 0');
            }])
            ->limit(10)
            ->get();

        $results = [];
        foreach ($products as $product) {
            $results[] = [
                'id' => $product->id,
                'name' => $product->name,
                'sale_price' => $product->sale_price,
                'stock_quantity' => $product->stock_quantity,
                'supplier_name' => $product->supplier->name ?? 'N/A',
                'batches' => $product->batches,
            ];
        }

        $this->productSearchResults = $results;
        $this->showProductSearchResults = count($results) > 0;
    }

    public function addProduct($productId, $name, $salePrice, $stockQuantity)
    {
        // Check if already added
        $exists = false;
        foreach ($this->selectedProducts as $product) {
            if ($product['product_id'] == $productId) {
                $exists = true;
                break;
            }
        }

        if (!$exists) {
            $this->selectedProducts[] = [
                'product_id' => $productId,
                'name' => $name,
                'quantity' => 1,
                'batch_id' => null,
                'sale_price' => $salePrice,
                'stock_quantity' => $stockQuantity,
            ];
        }

        $this->productSearchQuery = '';
        $this->productSearchResults = [];
        $this->showProductSearchResults = false;
        $this->closeProductModal();
    }

    public function removeProduct($index)
    {
        if (isset($this->selectedProducts[$index])) {
            $product = $this->selectedProducts[$index];
            if (isset($this->selectedProductBatches[$product['product_id']])) {
                unset($this->selectedProductBatches[$product['product_id']]);
            }
            unset($this->selectedProducts[$index]);
            $this->selectedProducts = array_values($this->selectedProducts);
        }
    }

    public function updateProductQuantity($index, $quantity)
    {
        $quantity = (int) $quantity;
        if (isset($this->selectedProducts[$index]) && $quantity > 0) {
            $maxQuantity = $this->selectedProducts[$index]['stock_quantity'];
            $this->selectedProducts[$index]['quantity'] = min($quantity, $maxQuantity);
        }
    }

    public function updateProductBatch($index, $batchId)
    {
        if (isset($this->selectedProducts[$index])) {
            $productId = $this->selectedProducts[$index]['product_id'];
            if ($batchId) {
                $this->selectedProductBatches[$productId] = $batchId;
                $this->selectedProducts[$index]['batch_id'] = $batchId;
            } else {
                unset($this->selectedProductBatches[$productId]);
                $this->selectedProducts[$index]['batch_id'] = null;
            }
        }
    }

    public function prepareReview()
    {
        // This method is called before step 3 to ensure data is ready
        // The view will compute the review data directly
    }

    public function save($redirectToPrint = false)
    {
        // Validate all steps
        $this->validateStep1();
        if (!$this->validateStep2()) {
            return;
        }

        // Prepare review items
        $this->prepareReview();

        DB::beginTransaction();
        try {
            // Use existing patient or create new one
            if ($this->isEditingPatient && $this->selectedPatientId) {
                $patient = Patient::findOrFail($this->selectedPatientId);
                // Update patient information if changed
                $patient->update([
                    'name' => trim($this->name),
                    'phone' => $this->phone ? trim($this->phone) : null,
                    'address' => $this->address ? trim($this->address) : null,
                ]);
            } else {
                // Create new patient with auto-generated code
                $patient = Patient::create([
                    'code' => Patient::generatePatientCode(),
                    'name' => trim($this->name),
                    'phone' => $this->phone ? trim($this->phone) : null,
                    'address' => $this->address ? trim($this->address) : null,
                    'clinic_id' => null,
                ]);
            }

            // Create invoice
            $invoice = Invoice::create([
                'invoice_number' => Invoice::generateInvoiceNumber(),
                'patient_id' => $patient->id,
                'clinic_id' => null, // Clinic removed from UI
                'user_id' => Auth::id(),
                'invoice_date' => $this->invoice_date,
                'notes' => $this->notes,
                'status' => 'completed',
                'total_amount' => 0, // Will be calculated
            ]);

            // Create invoice items from selected services
            foreach ($this->selectedServices as $service) {
                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'subcategory_id' => $service['subcategory_id'],
                    'product_id' => null,
                    'product_batch_id' => null,
                    'quantity' => 1,
                    'unit_price' => $service['cost'],
                    'total_price' => $service['cost'],
                    'notes' => null,
                ]);
            }

            // Create invoice items from selected products (drugs)
            foreach ($this->selectedProducts as $productData) {
                $productId = $productData['product_id'];
                $quantity = $productData['quantity'];
                $batchId = $productData['batch_id'] ?? null;
                
                $product = Product::find($productId);
                if ($product && $quantity > 0) {
                    InvoiceItem::create([
                        'invoice_id' => $invoice->id,
                        'subcategory_id' => null,
                        'product_id' => $productId,
                        'product_batch_id' => $batchId,
                        'quantity' => $quantity,
                        'unit_price' => $product->sale_price,
                        'total_price' => $product->sale_price * $quantity,
                        'notes' => null,
                    ]);

                    // Update product batch if selected
                    if ($batchId) {
                        $batch = ProductBatch::find($batchId);
                        if ($batch) {
                            $batch->sold_quantity += $quantity;
                            $batch->save();
                        }
                    }

                    // Update product stock
                    $product->stock_quantity = max(0, $product->stock_quantity - $quantity);
                    $product->save();
                }
            }

            // Calculate and update invoice total
            $invoice->calculateTotal();

            DB::commit();

            $this->alert('success', 'Patient and invoice created successfully!');
            
            if ($redirectToPrint) {
                return redirect()->route('invoices.show', $invoice);
            } else {
                // Reset form
                $this->resetForm();
                return;
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('error', 'Failed to create patient and invoice: ' . $e->getMessage());
        }
    }

    public function saveAndPrint()
    {
        return $this->save(true);
    }

    private function resetForm()
    {
        $this->name = '';
        $this->phone = '';
        $this->address = '';
        $this->notes = '';
        $this->selectedServices = [];
        $this->selectedProducts = [];
        $this->selectedProductBatches = [];
        $this->selectedPatientId = null;
        $this->isEditingPatient = false;
        $this->searchQuery = '';
        $this->searchResults = [];
        $this->showSearchResults = false;
    }

    public function render()
    {
        // Get batches for selected products
        $productBatchesMap = [];
        foreach ($this->selectedProducts as $product) {
            $productId = $product['product_id'];
            $productModel = Product::with(['batches' => function($query) {
                $query->whereRaw('quantity - sold_quantity + returned_quantity > 0');
            }])->find($productId);
            
            if ($productModel) {
                $productBatchesMap[$productId] = $productModel->batches;
            }
        }

        return view('livewire.patient-invoice-wizard', [
            'productBatchesMap' => $productBatchesMap,
        ]);
    }
}
