<?php

namespace App\Livewire;

use App\Models\Invoice;
use App\Models\Patient;
use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Facades\Auth;

class InvoiceList extends Component
{
    use LivewireAlert, WithPagination;

    protected $paginationTheme = 'bootstrap';
    
    public $search = '';
    public $startDate = '';
    public $endDate = '';
    public $status = '';
    public $patient_id = '';
    public $patientSearch = '';
    public $user_id = '';
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';

    protected $queryString = [
        'search' => ['except' => ''],
        'startDate' => ['except' => ''],
        'endDate' => ['except' => ''],
        'status' => ['except' => ''],
        'patient_id' => ['except' => ''],
        'user_id' => ['except' => ''],
    ];

    public function mount()
    {
        // Optional: Set default date range if needed
        
        // Check for session errors and display them
        if (session()->has('errors') && session('errors')->has('error')) {
            $this->alert('error', session('errors')->first('error'));
        }
        
        if (session('success')) {
            $this->alert('success', session('success'));
        }
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedStartDate()
    {
        $this->resetPage();
    }

    public function updatedEndDate()
    {
        $this->resetPage();
    }

    public function updatedStatus()
    {
        $this->resetPage();
    }

    public function updatedPatientId()
    {
        $this->resetPage();
    }

    public function updatedUserId()
    {
        $this->resetPage();
    }

    public function sortByField($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function deleteInvoice($invoiceId)
    {
        try {
            $invoice = Invoice::findOrFail($invoiceId);
            
            // Check for related invoice items
            if ($invoice->items()->count() > 0) {
                $this->alert('error', 'Cannot delete invoice. This invoice has associated items.');
                return;
            }

            $invoice->delete();
            $this->alert('success', 'Invoice deleted successfully!');
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                $this->alert('error', 'Cannot delete invoice. This invoice has related data that prevents deletion.');
            } else {
                $this->alert('error', 'An error occurred while deleting the invoice. Please try again.');
            }
        } catch (\Exception $e) {
            $this->alert('error', 'An unexpected error occurred. Please try again.');
        }
    }

    public function selectPatient($patientId)
    {
        $this->patient_id = $patientId;
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->patientSearch = $patient->name . ' (' . $patient->code . ')';
        }
        $this->resetPage();
    }
    
    public function updatedPatientSearch()
    {
        // Clear patient_id if search is cleared
        if (empty($this->patientSearch)) {
            $this->patient_id = '';
        }
    }

    public function clearPatient()
    {
        $this->patient_id = '';
        $this->patientSearch = '';
        $this->resetPage();
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->startDate = '';
        $this->endDate = '';
        $this->status = '';
        $this->patient_id = '';
        $this->patientSearch = '';
        $this->user_id = '';
        $this->sortBy = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }

    public function render()
    {
        $query = Invoice::with(['patient', 'user']);

        // Search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('invoice_number', 'like', '%' . $this->search . '%')
                  ->orWhere('notes', 'like', '%' . $this->search . '%')
                  ->orWhereHas('patient', function($patientQuery) {
                      $patientQuery->where('name', 'like', '%' . $this->search . '%')
                                   ->orWhere('code', 'like', '%' . $this->search . '%');
                  });
            });
        }

        // Date filter
        if ($this->startDate) {
            $query->whereDate('invoice_date', '>=', $this->startDate);
        }

        if ($this->endDate) {
            $query->whereDate('invoice_date', '<=', $this->endDate);
        }

        // Status filter
        if ($this->status) {
            $query->where('status', $this->status);
        }

        // Patient filter
        if ($this->patient_id) {
            $query->where('patient_id', $this->patient_id);
        }

        // User filter
        if ($this->user_id) {
            $query->where('user_id', $this->user_id);
        }

        // Sorting
        $query->orderBy($this->sortBy, $this->sortDirection);

        // Get filtered patients for search dropdown
        $filteredPatients = Patient::query();
        if ($this->patientSearch) {
            $filteredPatients->where(function($q) {
                $q->where('name', 'like', '%' . $this->patientSearch . '%')
                  ->orWhere('code', 'like', '%' . $this->patientSearch . '%')
                  ->orWhere('phone', 'like', '%' . $this->patientSearch . '%');
            });
        }
        $filteredPatients = $filteredPatients->orderBy('name')->limit(10)->get();

        $users = User::orderBy('name')->get();

        return view('livewire.invoice-list', [
            'invoices' => $query->paginate(50),
            'filteredPatients' => $filteredPatients,
            'users' => $users,
        ]);
    }
}

