<?php

namespace App\Livewire;

use App\Models\Patient;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Product;
use App\Models\ProductBatch;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use App\Traits\HasUtf8Encoding;

class EditInvoiceWizard extends Component
{
    use LivewireAlert, HasUtf8Encoding;

    public $invoiceId;
    public $invoice;

    // Modal states
    public $showServiceModal = false;
    public $showProductModal = false;

    // Patient form properties
    public $code, $name, $phone, $address;
    public $searchQuery = '';
    public $searchResults = [];
    public $showSearchResults = false;
    public $selectedPatientId = null;
    public $isEditingPatient = false;

    // Invoice form properties
    public $invoice_date, $notes;
    
    // Search properties for services
    public $serviceSearchQuery = '';
    public $serviceSearchResults = [];
    public $showServiceSearchResults = false;
    
    // Search properties for products (drugs)
    public $productSearchQuery = '';
    public $productSearchResults = [];
    public $showProductSearchResults = false;
    
    // Selected items (using list format)
    public $selectedServices = []; // Array of ['subcategory_id', 'category_id', 'name', 'cost', 'item_id']
    public $selectedProducts = []; // Array of ['product_id', 'name', 'quantity', 'batch_id', 'sale_price', 'item_id', 'stock_quantity']
    public $selectedProductBatches = []; // Array of [product_id => batch_id]
    
    // Available data
    public $categories = [];
    public $products = [];

    public function mount($invoice)
    {
        $this->invoiceId = $invoice->id;
        $this->invoice = Invoice::with(['patient', 'items.subcategory', 'items.product', 'items.productBatch'])->findOrFail($invoice->id);
        
        $this->categories = Category::with('subcategories')->get();
        $this->products = Product::with('supplier', 'batches')->get();
        
        // Load patient data
        $patient = $this->invoice->patient;
        if ($patient) {
            $this->selectedPatientId = $patient->id;
            $this->code = $patient->code;
            $this->name = $patient->name;
            $this->phone = $patient->phone;
            $this->address = $patient->address;
            $this->isEditingPatient = true;
        }
        
        // Load invoice data
        $this->invoice_date = $this->invoice->invoice_date->format('Y-m-d');
        $this->notes = $this->invoice->notes;
        
        // Load invoice items
        foreach ($this->invoice->items as $item) {
            if ($item->subcategory_id) {
                // Service
                $this->selectedServices[] = [
                    'subcategory_id' => $item->subcategory_id,
                    'category_id' => $item->subcategory->category_id ?? null,
                    'name' => $item->subcategory->name ?? 'N/A',
                    'cost' => $item->unit_price,
                    'item_id' => $item->id,
                ];
            } elseif ($item->product_id) {
                // Product/Drug
                $product = $item->product;
                // Stock quantity should include the quantity already in this invoice item
                $availableStock = ($product->stock_quantity ?? 0) + $item->quantity;
                $this->selectedProducts[] = [
                    'product_id' => $item->product_id,
                    'name' => $product->name ?? 'N/A',
                    'quantity' => $item->quantity,
                    'batch_id' => $item->product_batch_id,
                    'sale_price' => $item->unit_price,
                    'item_id' => $item->id,
                    'stock_quantity' => $availableStock,
                ];
            }
        }
    }

    public function updatedCode($value)
    {
        if ($this->isEditingPatient && $this->selectedPatientId) {
            return;
        }
        
        $this->searchQuery = $value;
        if (strlen($value) >= 1) {
            $this->searchPatients('code', $value);
        } else {
            $this->searchResults = [];
            $this->showSearchResults = false;
        }
    }

    public function updatedName($value)
    {
        if ($this->isEditingPatient && $this->selectedPatientId) {
            return;
        }
        
        $this->searchQuery = $value;
        if (strlen($value) >= 1) {
            $this->searchPatients('name', $value);
        } else {
            $this->searchResults = [];
            $this->showSearchResults = false;
        }
    }

    public function searchPatients($field = null, $value = null)
    {
        $searchValue = $value ?? $this->searchQuery;
        
        if (empty($searchValue)) {
            $this->searchResults = [];
            $this->showSearchResults = false;
            return;
        }

        $query = Patient::query();

        if ($field === 'code' || (!$field && strlen($searchValue) >= 1)) {
            $query->where('code', 'like', '%' . $searchValue . '%');
        }

        if ($field === 'name' || (!$field && strlen($searchValue) >= 1)) {
            if ($field === 'name') {
                $query->where('name', 'like', '%' . $searchValue . '%');
            } else {
                $query->orWhere('name', 'like', '%' . $searchValue . '%');
            }
        }

        $this->searchResults = $query->limit(10)->get();
        $this->showSearchResults = count($this->searchResults) > 0;
    }

    public function selectPatient($patientId)
    {
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->selectedPatientId = $patient->id;
            $this->code = $patient->code;
            $this->name = $patient->name;
            $this->phone = $patient->phone;
            $this->address = $patient->address;
            $this->isEditingPatient = true;
            $this->searchResults = [];
            $this->showSearchResults = false;
            $this->searchQuery = '';
        }
    }

    public function clearPatientSelection()
    {
        $this->selectedPatientId = null;
        $this->isEditingPatient = false;
        $this->code = '';
        $this->name = '';
        $this->phone = '';
        $this->address = '';
        $this->searchQuery = '';
        $this->searchResults = [];
        $this->showSearchResults = false;
    }

    // Service search and management
    public function updatedServiceSearchQuery($value)
    {
        if (strlen($value) >= 1) {
            $this->searchServices($value);
        } else {
            $this->serviceSearchResults = [];
            $this->showServiceSearchResults = false;
        }
    }

    public function searchServices($query = null)
    {
        $searchValue = $query ?? $this->serviceSearchQuery;
        
        if (empty($searchValue)) {
            $this->serviceSearchResults = [];
            $this->showServiceSearchResults = false;
            return;
        }

        $serviceCategories = Category::whereIn('type', ['service', 'other'])
            ->with(['subcategories' => function($q) use ($searchValue) {
                $q->where('name', 'like', '%' . $searchValue . '%');
            }])
            ->get();

        $results = [];
        foreach ($serviceCategories as $category) {
            foreach ($category->subcategories as $subcategory) {
                $results[] = [
                    'id' => $subcategory->id,
                    'category_id' => $category->id,
                    'category_name' => $category->name,
                    'name' => $subcategory->name,
                    'cost' => $subcategory->cost,
                ];
            }
        }

        $this->serviceSearchResults = $results;
        $this->showServiceSearchResults = count($results) > 0;
    }

    public function addService($subcategoryId, $categoryId, $name, $cost)
    {
        // Check if already added
        $exists = false;
        foreach ($this->selectedServices as $service) {
            if ($service['subcategory_id'] == $subcategoryId) {
                $exists = true;
                break;
            }
        }

        if (!$exists) {
            $this->selectedServices[] = [
                'subcategory_id' => $subcategoryId,
                'category_id' => $categoryId,
                'name' => $name,
                'cost' => $cost,
                'item_id' => null, // New item
            ];
        }

        $this->serviceSearchQuery = '';
        $this->serviceSearchResults = [];
        $this->showServiceSearchResults = false;
        $this->closeServiceModal();
    }

    public function removeService($index)
    {
        if (isset($this->selectedServices[$index])) {
            unset($this->selectedServices[$index]);
            $this->selectedServices = array_values($this->selectedServices);
        }
    }

    // Product (drug) search and management
    public function updatedProductSearchQuery($value)
    {
        if (strlen($value) >= 1) {
            $this->searchProducts($value);
        } else {
            $this->productSearchResults = [];
            $this->showProductSearchResults = false;
        }
    }

    public function searchProducts($query = null)
    {
        $searchValue = $query ?? $this->productSearchQuery;
        
        if (empty($searchValue)) {
            $this->productSearchResults = [];
            $this->showProductSearchResults = false;
            return;
        }

        $products = Product::where('name', 'like', '%' . $searchValue . '%')
            ->where('stock_quantity', '>', 0)
            ->with(['supplier', 'batches' => function($q) {
                $q->whereRaw('quantity - sold_quantity + returned_quantity > 0');
            }])
            ->limit(10)
            ->get();

        $results = [];
        foreach ($products as $product) {
            $results[] = [
                'id' => $product->id,
                'name' => $product->name,
                'sale_price' => $product->sale_price,
                'stock_quantity' => $product->stock_quantity,
                'supplier_name' => $product->supplier->name ?? 'N/A',
                'batches' => $product->batches,
            ];
        }

        $this->productSearchResults = $results;
        $this->showProductSearchResults = count($results) > 0;
    }

    public function addProduct($productId, $name, $salePrice, $stockQuantity)
    {
        // Check if already added
        $exists = false;
        foreach ($this->selectedProducts as $product) {
            if ($product['product_id'] == $productId) {
                $exists = true;
                break;
            }
        }

        if (!$exists) {
            $this->selectedProducts[] = [
                'product_id' => $productId,
                'name' => $name,
                'quantity' => 1,
                'batch_id' => null,
                'sale_price' => $salePrice,
                'item_id' => null, // New item
                'stock_quantity' => $stockQuantity,
            ];
        }

        $this->productSearchQuery = '';
        $this->productSearchResults = [];
        $this->showProductSearchResults = false;
        $this->closeProductModal();
    }

    public function removeProduct($index)
    {
        if (isset($this->selectedProducts[$index])) {
            $product = $this->selectedProducts[$index];
            if (isset($this->selectedProductBatches[$product['product_id']])) {
                unset($this->selectedProductBatches[$product['product_id']]);
            }
            unset($this->selectedProducts[$index]);
            $this->selectedProducts = array_values($this->selectedProducts);
        }
    }

    public function updateProductQuantity($index, $quantity)
    {
        $quantity = (int) $quantity;
        if (isset($this->selectedProducts[$index]) && $quantity > 0) {
            $maxQuantity = $this->selectedProducts[$index]['stock_quantity'];
            $this->selectedProducts[$index]['quantity'] = min($quantity, $maxQuantity);
        }
    }

    public function updateProductBatch($index, $batchId)
    {
        if (isset($this->selectedProducts[$index])) {
            $productId = $this->selectedProducts[$index]['product_id'];
            if ($batchId) {
                $this->selectedProductBatches[$productId] = $batchId;
                $this->selectedProducts[$index]['batch_id'] = $batchId;
            } else {
                unset($this->selectedProductBatches[$productId]);
                $this->selectedProducts[$index]['batch_id'] = null;
            }
        }
    }

    public function openServiceModal()
    {
        $this->showServiceModal = true;
        $this->serviceSearchQuery = '';
        $this->serviceSearchResults = [];
        $this->showServiceSearchResults = false;
    }

    public function closeServiceModal()
    {
        $this->showServiceModal = false;
        $this->serviceSearchQuery = '';
        $this->serviceSearchResults = [];
        $this->showServiceSearchResults = false;
    }

    public function openProductModal()
    {
        $this->showProductModal = true;
        $this->productSearchQuery = '';
        $this->productSearchResults = [];
        $this->showProductSearchResults = false;
    }

    public function closeProductModal()
    {
        $this->showProductModal = false;
        $this->productSearchQuery = '';
        $this->productSearchResults = [];
        $this->showProductSearchResults = false;
    }

    private function validateStep1()
    {
        if ($this->isEditingPatient && $this->selectedPatientId) {
            $this->validate([
                'selectedPatientId' => 'required|exists:patients,id',
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:255',
                'address' => 'nullable|string',
            ], [
                'selectedPatientId.required' => 'Please select a patient.',
                'selectedPatientId.exists' => 'Selected patient does not exist.',
                'name.required' => 'Patient name is required.',
                'phone.required' => 'Phone number is required.',
            ]);
        } else {
            $this->validate([
                'code' => 'required|string|max:255|unique:patients,code',
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:255',
                'address' => 'nullable|string',
            ], [
                'code.required' => 'Patient code is required.',
                'code.unique' => 'This patient code already exists. Please select from search results or use a different code.',
                'name.required' => 'Patient name is required.',
                'phone.required' => 'Phone number is required.',
            ]);
        }
    }

    private function validateStep2()
    {
        $hasServices = !empty($this->selectedServices);
        $hasDrugs = !empty($this->selectedProducts);
        
        if (!$hasServices && !$hasDrugs) {
            $this->addError('items', 'Please add at least one service or product (drug) for the invoice.');
            return false;
        }
        
        foreach ($this->selectedProducts as $index => $product) {
            if ($product['quantity'] <= 0 || $product['quantity'] > $product['stock_quantity']) {
                $this->addError('items', 'Invalid quantity for product: ' . $product['name']);
                return false;
            }
        }
        
        return true;
    }

    public function save()
    {
        $this->validateStep1();
        if (!$this->validateStep2()) {
            return;
        }

        DB::beginTransaction();
        try {
            // Update or create patient
            if ($this->isEditingPatient && $this->selectedPatientId) {
                $patient = Patient::findOrFail($this->selectedPatientId);
                $patient->update([
                    'name' => trim($this->name),
                    'phone' => trim($this->phone),
                    'address' => $this->address ? trim($this->address) : null,
                ]);
            } else {
                $patient = Patient::create([
                    'code' => trim($this->code),
                    'name' => trim($this->name),
                    'phone' => trim($this->phone),
                    'address' => $this->address ? trim($this->address) : null,
                    'clinic_id' => null,
                ]);
            }

            // Update invoice
            $invoice = Invoice::findOrFail($this->invoiceId);
            $invoice->update([
                'patient_id' => $patient->id,
                'invoice_date' => $this->invoice_date,
                'notes' => $this->notes,
            ]);

            // Get existing item IDs to track what to delete
            $existingServiceItemIds = [];
            $existingProductItemIds = [];
            
            foreach ($this->selectedServices as $service) {
                if (isset($service['item_id']) && $service['item_id']) {
                    $existingServiceItemIds[] = $service['item_id'];
                }
            }
            
            foreach ($this->selectedProducts as $product) {
                if (isset($product['item_id']) && $product['item_id']) {
                    $existingProductItemIds[] = $product['item_id'];
                }
            }

            // Delete removed items and restore stock
            $allExistingItemIds = array_merge($existingServiceItemIds, $existingProductItemIds);
            $itemsToDelete = InvoiceItem::where('invoice_id', $invoice->id)
                ->whereNotIn('id', $allExistingItemIds)
                ->get();
            
            foreach ($itemsToDelete as $item) {
                // Restore product stock if it's a product item
                if ($item->product_id) {
                    if ($item->product_batch_id) {
                        $batch = ProductBatch::find($item->product_batch_id);
                        if ($batch) {
                            $batch->sold_quantity = max(0, $batch->sold_quantity - $item->quantity);
                            $batch->save();
                        }
                    }
                    
                    $product = Product::find($item->product_id);
                    if ($product) {
                        $product->stock_quantity += $item->quantity;
                        $product->save();
                    }
                }
                $item->delete();
            }

            // Update or create service items
            foreach ($this->selectedServices as $service) {
                if (isset($service['item_id']) && $service['item_id']) {
                    // Update existing item
                    InvoiceItem::where('id', $service['item_id'])->update([
                        'subcategory_id' => $service['subcategory_id'],
                        'unit_price' => $service['cost'],
                        'total_price' => $service['cost'],
                    ]);
                } else {
                    // Create new item
                    InvoiceItem::create([
                        'invoice_id' => $invoice->id,
                        'subcategory_id' => $service['subcategory_id'],
                        'product_id' => null,
                        'product_batch_id' => null,
                        'quantity' => 1,
                        'unit_price' => $service['cost'],
                        'total_price' => $service['cost'],
                        'notes' => null,
                    ]);
                }
            }

            // Update or create product items
            foreach ($this->selectedProducts as $productData) {
                $productId = $productData['product_id'];
                $quantity = $productData['quantity'];
                $batchId = $productData['batch_id'] ?? null;
                
                $product = Product::find($productId);
                if ($product && $quantity > 0) {
                    if (isset($productData['item_id']) && $productData['item_id']) {
                        // Update existing item - need to restore old stock first
                        $oldItem = InvoiceItem::find($productData['item_id']);
                        if ($oldItem) {
                            $oldQuantity = $oldItem->quantity;
                            $quantityDiff = $quantity - $oldQuantity;
                            
                            // Restore old quantity from batch
                            if ($oldItem->product_batch_id) {
                                $oldBatch = ProductBatch::find($oldItem->product_batch_id);
                                if ($oldBatch) {
                                    $oldBatch->sold_quantity = max(0, $oldBatch->sold_quantity - $oldQuantity);
                                    $oldBatch->save();
                                }
                            }
                            
                            // Restore old quantity from product stock
                            $oldProduct = Product::find($oldItem->product_id);
                            if ($oldProduct) {
                                $oldProduct->stock_quantity += $oldQuantity;
                                $oldProduct->save();
                            }
                            
                            // Update item
                            $oldItem->update([
                                'quantity' => $quantity,
                                'unit_price' => $product->sale_price,
                                'total_price' => $product->sale_price * $quantity,
                                'product_batch_id' => $batchId,
                            ]);
                            
                            // Deduct new quantity (only the difference if increased)
                            if ($quantityDiff > 0) {
                                // Update product batch if selected
                                if ($batchId) {
                                    $batch = ProductBatch::find($batchId);
                                    if ($batch) {
                                        $batch->sold_quantity += $quantityDiff;
                                        $batch->save();
                                    }
                                }
                                
                                // Update product stock
                                $product->stock_quantity = max(0, $product->stock_quantity - $quantityDiff);
                                $product->save();
                            } elseif ($quantityDiff < 0) {
                                // Quantity decreased, stock already restored above
                                // Just update batch if changed
                                if ($batchId && $batchId != $oldItem->product_batch_id) {
                                    $batch = ProductBatch::find($batchId);
                                    if ($batch) {
                                        $batch->sold_quantity += $quantity;
                                        $batch->save();
                                    }
                                }
                            }
                        }
                    } else {
                        // Create new item
                        InvoiceItem::create([
                            'invoice_id' => $invoice->id,
                            'subcategory_id' => null,
                            'product_id' => $productId,
                            'product_batch_id' => $batchId,
                            'quantity' => $quantity,
                            'unit_price' => $product->sale_price,
                            'total_price' => $product->sale_price * $quantity,
                            'notes' => null,
                        ]);
                        
                        // Update product batch if selected
                        if ($batchId) {
                            $batch = ProductBatch::find($batchId);
                            if ($batch) {
                                $batch->sold_quantity += $quantity;
                                $batch->save();
                            }
                        }

                        // Update product stock
                        $product->stock_quantity = max(0, $product->stock_quantity - $quantity);
                        $product->save();
                    }
                }
            }

            // Calculate and update invoice total
            $invoice->calculateTotal();

            DB::commit();

            $this->alert('success', 'Invoice updated successfully!');
            return redirect()->route('invoices.show', $invoice);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('error', 'Failed to update invoice: ' . $e->getMessage());
        }
    }

    public function render()
    {
        // Get batches for selected products
        $productBatchesMap = [];
        foreach ($this->selectedProducts as $product) {
            $productId = $product['product_id'];
            $productModel = Product::with(['batches' => function($query) {
                $query->whereRaw('quantity - sold_quantity + returned_quantity > 0');
            }])->find($productId);
            
            if ($productModel) {
                $productBatchesMap[$productId] = $productModel->batches;
            }
        }

        return view('livewire.edit-invoice-wizard', [
            'productBatchesMap' => $productBatchesMap,
        ]);
    }
}

