<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\Currency;
use App\Models\User;
use App\Models\Notification;
use App\Services\OneSignalService;
use App\Services\NotificationTranslationService;
use Illuminate\Support\Facades\Log;

class SendCurrencyRateNotificationJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $currency;
    protected $field;
    protected $value;
    protected $oldRate;
    protected $updatedBy;

    /**
     * Create a new job instance.
     */
    public function __construct(Currency $currency, string $field, float $value, ?float $oldRate = null, ?string $updatedBy = null)
    {
        $this->currency = $currency;
        $this->field = $field;
        $this->value = $value;
        $this->oldRate = $oldRate;
        $this->updatedBy = $updatedBy;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            // Get users who have enabled currency rate notifications AND have currency rate access
            $users = User::where('notifications_enabled', true)
                         ->where('currency_rate_notifications', true)
                         ->where('currency_rate_access', true)
                         ->get(['id', 'language_preference']);
            
            if ($users->isEmpty()) {
                Log::info("No users have enabled currency rate notifications and access", [
                    'currency' => $this->currency->currency_name,
                    'field' => $this->field,
                    'value' => $this->value,
                    'filter_criteria' => 'notifications_enabled=true, currency_rate_notifications=true, currency_rate_access=true'
                ]);
                return;
            }
            
            // Initialize OneSignal service and translation service
            $oneSignalService = new OneSignalService();
            $translationService = new NotificationTranslationService();
            
            // Group users by language preference for efficient notification sending
            $usersByLanguage = $users->groupBy(function ($user) {
                return $user->language_preference ?? 'en';
            });
            
            $totalNotificationsSent = 0;
            
            // Send notifications for each language group
            foreach ($usersByLanguage as $language => $languageUsers) {
                $userIds = $languageUsers->pluck('id')->toArray();
                
                // Create notification message in user's preferred language
                $message = $translationService->getCurrencyRateMessage(
                    $this->currency, 
                    $this->value, 
                    $this->oldRate, 
                    $this->field, 
                    $language
                );
                
                // Send notification to users with this language preference
                $result = $oneSignalService->sendNotificationToUsers($userIds, $message, [
                    'type' => 'currency_rate_update',
                    'currency_name' => $this->currency->currency_name,
                    'currency_code' => $this->currency->code ?? 'unknown',
                    'field' => $this->field,
                    'value' => $this->value,
                    'old_rate' => $this->oldRate,
                    'updated_by' => $this->updatedBy ?? 'System',
                    'timestamp' => now()->toISOString(),
                ], $language);
                
                $totalNotificationsSent += $result['recipients'] ?? 0;
                
                Log::info("Currency rate notification sent for language group", [
                    'language' => $language,
                    'user_count' => count($userIds),
                    'success' => $result['success'] ?? false,
                    'recipients' => $result['recipients'] ?? 0,
                ]);
            }
            
            // Log the overall notification result
            Log::info("Currency rate notification sent via queue", [
                'currency' => $this->currency->currency_name,
                'field' => $this->field,
                'value' => $this->value,
                'old_rate' => $this->oldRate,
                'total_users' => $users->count(),
                'total_notifications_sent' => $totalNotificationsSent,
                'language_groups' => $usersByLanguage->keys()->toArray(),
                'updated_by' => $this->updatedBy ?? 'System'
            ]);
            
        } catch (\Exception $e) {
            // Log error but don't fail the job
            Log::error("Failed to send currency rate notification via queue", [
                'currency' => $this->currency->currency_name,
                'field' => $this->field,
                'value' => $this->value,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
