<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\UserStoreRequest;
use App\Http\Requests\UserUpdateRequest;
use Illuminate\Support\Facades\Session;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', User::class);

        $search = $request->get('search', '');

        $users = User::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view('app.users.index', compact('users', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', User::class);

        $roles = Role::get();

        return view('app.users.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(UserStoreRequest $request): RedirectResponse
    {
        $this->authorize('create', User::class);

        try {
            $validated = $request->validated();

            $validated['password'] = Hash::make($validated['password']);

            $user = User::create($validated);

              // Sync roles, assuming roles come in as an array of IDs
              if ($request->has('roles') && is_array($request->roles) && count($request->roles) > 0) {
                $roles = Role::whereIn('id', $request->roles)->pluck('name')->toArray();
                $user->syncRoles($roles); // Sync multiple roles
            } else {
                // Default to 'user' role if no roles are provided
                $user->assignRole('user');
            }  
            Session::flash('message', [
                'message' => 'User Created Successfully',
                'type' => 'alert-success',
            ]);
            return redirect()
                ->route('users.index', $user)
                ->withSuccess(__('crud.common.created'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while creating the user. Please try again.']);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, User $user): View
    {
        $this->authorize('view', $user);

        return view('app.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, User $user): View
    {
        $this->authorize('update', $user);

        $roles = Role::get();

        return view('app.users.edit', compact('user', 'roles'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        UserUpdateRequest $request,
        User $user
    ): RedirectResponse {
        $this->authorize('update', $user);

        try {
            $validated = $request->validated();

            if (empty($validated['password'])) {
                unset($validated['password']);
            } else {
                $validated['password'] = Hash::make($validated['password']);
            }

            $user->update($validated);

                // Sync roles, assuming roles come in as an array of IDs
                if ($request->has('roles') && is_array($request->roles) && count($request->roles) > 0) {
                    $roles = Role::whereIn('id', $request->roles)->pluck('name')->toArray();
                    $user->syncRoles($roles); // Sync multiple roles
                } else {
                    // If no roles provided and user doesn't have any role, assign 'user' role
                    if ($user->roles->isEmpty()) {
                        $user->assignRole('user');
                    }
                }      
                Session::flash('message', [
                'message' => 'User Updated Successfully',
                'type' => 'alert-success',
            ]);
            return redirect()
                ->route('users.index', $user)
                ->withSuccess(__('crud.common.saved'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while updating the user. Please try again.']);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, User $user): RedirectResponse
    {
        $this->authorize('delete', $user);

        try {
            // Check for related data
            if ($user->invoices()->count() > 0) {
                return redirect()
                    ->route('users.index')
                    ->withErrors(['error' => 'Cannot delete user. This user has associated invoices.']);
            }

            if ($user->patientHistories()->count() > 0) {
                return redirect()
                    ->route('users.index')
                    ->withErrors(['error' => 'Cannot delete user. This user has associated patient histories.']);
            }

            $user->delete();
            
            Session::flash('message', [
                'message' => 'User Deleted Successfully',
                'type' => 'alert-success',
            ]);
            return redirect()
                ->route('users.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('users.index')
                    ->withErrors(['error' => 'Cannot delete user. This user has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('users.index')
                ->withErrors(['error' => 'An error occurred while deleting the user. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('users.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
