<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;

class SupplierController extends Controller
{
    public function index(Request $request): View
    {
        $this->authorize('view-any', Supplier::class);
        $search = $request->get('search', '');
        $suppliers = Supplier::search($search)->latest()->paginate(50)->withQueryString();
        return view('app.suppliers.index', compact('suppliers', 'search'));
    }

    public function create(Request $request): View
    {
        $this->authorize('create', Supplier::class);
        return view('app.suppliers.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Supplier::class);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
        ]);
        Supplier::create($validated);
        return redirect()->route('suppliers.index')->withSuccess(__('crud.common.created'));
    }

    public function show(Request $request, Supplier $supplier): View
    {
        $this->authorize('view', $supplier);
        $supplier->load('products');
        return view('app.suppliers.show', compact('supplier'));
    }

    public function edit(Request $request, Supplier $supplier): View
    {
        $this->authorize('update', $supplier);
        return view('app.suppliers.edit', compact('supplier'));
    }

    public function update(Request $request, Supplier $supplier): RedirectResponse
    {
        $this->authorize('update', $supplier);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
        ]);
        $supplier->update($validated);
        return redirect()->route('suppliers.index')->withSuccess(__('crud.common.saved'));
    }

    public function destroy(Request $request, Supplier $supplier): RedirectResponse
    {
        $this->authorize('delete', $supplier);
        
        try {
            // Check for related products
            if ($supplier->products()->count() > 0) {
                return redirect()
                    ->route('suppliers.index')
                    ->withErrors(['error' => 'Cannot delete supplier. This supplier has associated products.']);
            }

            $supplier->delete();
            return redirect()->route('suppliers.index')->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('suppliers.index')
                    ->withErrors(['error' => 'Cannot delete supplier. This supplier has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('suppliers.index')
                ->withErrors(['error' => 'An error occurred while deleting the supplier. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('suppliers.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}

