<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Supplier;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;

class ProductController extends Controller
{
    public function index(Request $request): View
    {
        $this->authorize('view-any', Product::class);
        $search = $request->get('search', '');
        $products = Product::with('supplier')->search($search)->latest()->paginate(50)->withQueryString();
        return view('app.products.index', compact('products', 'search'));
    }

    public function create(Request $request): View
    {
        $this->authorize('create', Product::class);
        $suppliers = Supplier::pluck('name', 'id');
        return view('app.products.create', compact('suppliers'));
    }

    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Product::class);
        
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'sale_price' => 'required|numeric|min:0',
                'purchase_price' => 'nullable|numeric|min:0',
                'supplier_id' => 'nullable|exists:suppliers,id',
                'expired_date' => 'nullable|date',
                'stock_quantity' => 'nullable|integer|min:0',
                'description' => 'nullable|string',
            ]);
            Product::create($validated);
            return redirect()->route('products.index')->withSuccess(__('crud.common.created'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while creating the product. Please try again.']);
        }
    }

    public function show(Request $request, Product $product): View
    {
        $this->authorize('view', $product);
        $product->load(['supplier', 'batches', 'invoiceItems.invoice.patient']);
        
        // Get invoices where this product was sold
        $invoiceItems = $product->invoiceItems()
            ->with(['invoice.patient'])
            ->latest()
            ->paginate(50);
        
        return view('app.products.show', compact('product', 'invoiceItems'));
    }

    public function edit(Request $request, Product $product): View
    {
        $this->authorize('update', $product);
        $suppliers = Supplier::pluck('name', 'id');
        return view('app.products.edit', compact('product', 'suppliers'));
    }

    public function update(Request $request, Product $product): RedirectResponse
    {
        $this->authorize('update', $product);
        
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'sale_price' => 'required|numeric|min:0',
                'purchase_price' => 'nullable|numeric|min:0',
                'supplier_id' => 'nullable|exists:suppliers,id',
                'expired_date' => 'nullable|date',
                'stock_quantity' => 'nullable|integer|min:0',
                'description' => 'nullable|string',
            ]);
            $product->update($validated);
            return redirect()->route('products.index')->withSuccess(__('crud.common.saved'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while updating the product. Please try again.']);
        }
    }

    public function destroy(Request $request, Product $product): RedirectResponse
    {
        $this->authorize('delete', $product);
        
        try {
            // Check for related data
            if ($product->invoiceItems()->count() > 0) {
                return redirect()
                    ->route('products.index')
                    ->withErrors(['error' => 'Cannot delete product. This product has been used in invoices.']);
            }

            if ($product->batches()->count() > 0) {
                return redirect()
                    ->route('products.index')
                    ->withErrors(['error' => 'Cannot delete product. This product has associated batches.']);
            }

            $product->delete();
            return redirect()->route('products.index')->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('products.index')
                    ->withErrors(['error' => 'Cannot delete product. This product has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('products.index')
                ->withErrors(['error' => 'An error occurred while deleting the product. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('products.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}

