<?php

namespace App\Http\Controllers;

use App\Models\Patient;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;

class PatientController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Patient::class);

        $search = $request->get('search', '');

        $patients = Patient::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view('app.patients.index', compact('patients', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Patient::class);

        return view('app.patients.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Patient::class);

        try {
            $validated = $request->validate([
                'code' => 'required|string|max:255|unique:patients,code',
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:255',
                'address' => 'nullable|string',
            ]);

            Patient::create($validated);

            return redirect()
                ->route('patients.index')
                ->withSuccess(__('crud.common.created'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while creating the patient. Please try again.']);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Patient $patient): View
    {
        $this->authorize('view', $patient);

        $patient->load([
            'invoices.items' => function($query) {
                $query->with(['subcategory.category', 'product', 'productBatch']);
            },
            'invoices' => function($query) {
                $query->latest()->take(50);
            },
            'histories' => function($query) {
                $query->latest()->take(50);
            }
        ]);

        // Get all invoice items for this patient (services and drugs)
        $invoiceItems = \App\Models\InvoiceItem::whereHas('invoice', function($query) use ($patient) {
            $query->where('patient_id', $patient->id);
        })
        ->with(['invoice', 'subcategory.category', 'product', 'productBatch'])
        ->orderBy('created_at', 'desc')
        ->get();

        return view('app.patients.show', compact('patient', 'invoiceItems'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Patient $patient): View
    {
        $this->authorize('update', $patient);

        return view('app.patients.edit', compact('patient'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Patient $patient): RedirectResponse
    {
        $this->authorize('update', $patient);

        try {
            $validated = $request->validate([
                'code' => 'required|string|max:255|unique:patients,code,' . $patient->id,
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:255',
                'address' => 'nullable|string',
            ]);

            $patient->update($validated);

            return redirect()
                ->route('patients.show', $patient)
                ->withSuccess(__('crud.common.saved'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while updating the patient. Please try again.']);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Patient $patient): RedirectResponse
    {
        $this->authorize('delete', $patient);

        try {
            // Check for related data
            if ($patient->invoices()->count() > 0) {
                return redirect()
                    ->route('patients.index')
                    ->withErrors(['error' => 'Cannot delete patient. This patient has associated invoices.']);
            }

            if ($patient->histories()->count() > 0) {
                return redirect()
                    ->route('patients.index')
                    ->withErrors(['error' => 'Cannot delete patient. This patient has associated histories.']);
            }

            $patient->delete();

            return redirect()
                ->route('patients.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('patients.index')
                    ->withErrors(['error' => 'Cannot delete patient. This patient has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('patients.index')
                ->withErrors(['error' => 'An error occurred while deleting the patient. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('patients.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}

