<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Patient;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;

class InvoiceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Invoice::class);

        return view('app.invoices.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Invoice::class);

        $patients = Patient::all();

        return view('app.invoices.create', compact('patients'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Invoice::class);

        $validated = $request->validate([
            'patient_id' => 'required|exists:patients,id',
            'invoice_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        $validated['invoice_number'] = Invoice::generateInvoiceNumber();
        $validated['user_id'] = Auth::id();
        $validated['status'] = 'draft';
        $validated['total_amount'] = 0;

        $invoice = Invoice::create($validated);

        return redirect()
            ->route('invoices.edit', $invoice)
            ->withSuccess(__('crud.common.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Invoice $invoice): View
    {
        $this->authorize('view', $invoice);

        $invoice->load(['patient', 'user', 'items.subcategory', 'items.product', 'items.productBatch']);

        return view('app.invoices.show', compact('invoice'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Invoice $invoice): View
    {
        $this->authorize('update', $invoice);

        $invoice->load(['items.subcategory', 'items.product', 'items.productBatch']);
        $patients = Patient::all();

        return view('app.invoices.edit', compact('invoice', 'patients'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Invoice $invoice): RedirectResponse
    {
        $this->authorize('update', $invoice);

        $validated = $request->validate([
            'patient_id' => 'required|exists:patients,id',
            'invoice_date' => 'required|date',
            'notes' => 'nullable|string',
            'status' => 'required|in:draft,completed,cancelled',
        ]);

        $invoice->update($validated);
        $invoice->calculateTotal();

        return redirect()
            ->route('invoices.index')
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Invoice $invoice): RedirectResponse
    {
        $this->authorize('delete', $invoice);

        try {
            // Check for related invoice items
            if ($invoice->items()->count() > 0) {
                return redirect()
                    ->route('invoices.index')
                    ->withErrors(['error' => 'Cannot delete invoice. This invoice has associated items.']);
            }

            $invoice->delete();

            return redirect()
                ->route('invoices.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('invoices.index')
                    ->withErrors(['error' => 'Cannot delete invoice. This invoice has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('invoices.index')
                ->withErrors(['error' => 'An error occurred while deleting the invoice. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('invoices.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
