<?php

namespace App\Http\Controllers;

use App\Models\Clinic;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;

class ClinicController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Clinic::class);

        $search = $request->get('search', '');

        $clinics = Clinic::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view('app.clinics.index', compact('clinics', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Clinic::class);

        return view('app.clinics.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Clinic::class);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
        ]);

        Clinic::create($validated);

        return redirect()
            ->route('clinics.index')
            ->withSuccess(__('crud.common.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Clinic $clinic): View
    {
        $this->authorize('view', $clinic);

        $clinic->load(['patients', 'invoices']);

        return view('app.clinics.show', compact('clinic'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Clinic $clinic): View
    {
        $this->authorize('update', $clinic);

        return view('app.clinics.edit', compact('clinic'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Clinic $clinic): RedirectResponse
    {
        $this->authorize('update', $clinic);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
        ]);

        $clinic->update($validated);

        return redirect()
            ->route('clinics.index')
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Clinic $clinic): RedirectResponse
    {
        $this->authorize('delete', $clinic);

        try {
            // Check for related data
            if ($clinic->patients()->count() > 0) {
                return redirect()
                    ->route('clinics.index')
                    ->withErrors(['error' => 'Cannot delete clinic. This clinic has associated patients.']);
            }

            if ($clinic->invoices()->count() > 0) {
                return redirect()
                    ->route('clinics.index')
                    ->withErrors(['error' => 'Cannot delete clinic. This clinic has associated invoices.']);
            }

            $clinic->delete();

            return redirect()
                ->route('clinics.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('clinics.index')
                    ->withErrors(['error' => 'Cannot delete clinic. This clinic has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('clinics.index')
                ->withErrors(['error' => 'An error occurred while deleting the clinic. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('clinics.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}

