<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\AccountTransactionResource;
use App\Http\Resources\AccountTransactionCollection;
use App\Services\TransactionNotificationService;

class UserAccountTransactionsController extends Controller
{
    protected $notificationService;

    public function __construct(TransactionNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    public function index(
        Request $request,
        User $user
    ): AccountTransactionCollection {
        $this->authorize('view', $user);

        $search = $request->get('search', '');

        $accountTransactions = $user
            ->accountTransactions()
            ->search($search)
            ->latest()
            ->paginate();

        return new AccountTransactionCollection($accountTransactions);
    }

    public function store(
        Request $request,
        User $user
    ): AccountTransactionResource {
        $this->authorize('create', AccountTransaction::class);

        $validated = $request->validate([
            'account_id' => ['required', 'exists:accounts,id'],
            'transaction_type' => [
                'required',
                'in:deposit,withdrawal,transfer',
            ],
            'transaction_amount' => ['required', 'numeric'],
            'transaction_date' => ['required', 'date'],
            'transaction_time' => ['required', 'date_format:H:i:s'],
            'note' => ['required', 'max:255', 'string'],
        ]);

        $accountTransaction = $user->accountTransactions()->create($validated);

        // Send notification for new transaction
        $this->notificationService->sendTransactionCreatedNotification($accountTransaction);

        return new AccountTransactionResource($accountTransaction);
    }
}
