<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AccountTransaction;
use App\Models\Account;
use App\Models\Customer;
use App\Models\TransactionFee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Barryvdh\DomPDF\PDF;

class TransactionController extends Controller
{
    /**
     * Get transactions (modified to work like web interface)
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Check if account_id is provided in request (for specific account)
        $accountId = $request->get('account_id');
        
        if ($accountId) {
            // If account_id is provided, show transactions for that specific account (like web interface)
            $account = Account::find($accountId);
            
            if (!$account) {
                return response()->json([
                    'success' => false,
                    'message' => 'Account not found'
                ], 404);
            }

            // Use the same logic as the Livewire TransactionsAccount component
            // This ensures we only return transactions that affect the customer
            $query = AccountTransaction::with([
                'from_account.customer:id,customer_name',
                'from_account.currency:id,currency_name',
                'to_account.customer:id,customer_name',
                'to_account.currency:id,currency_name',
                'fees:id,fee_amount,feeable_id,feeable_type'
            ])
            ->select([
                'id', 'transaction_type', 'transaction_amount', 
                DB::raw('CAST(DATE(created_at) AS CHAR) as transaction_date'),
                'transaction_time', 'person_name', 'note', 'receiver_name', 'from_account_id', 
                'to_account_id', 'transaction_group_id', 'status', 'created_at', 'get_fee'
            ])
            ->where('transaction_type', '!=', 'fee') // Exclude fee transactions
            ->where('to_account_id', '!=', $accountId) // Exclude transactions TO this account
            ->whereNotIn('from_account_id', [1]) // Exclude transactions FROM customer ID 1 (cashbox)
            ->where('from_account_id', $accountId); // Only transactions FROM the customer's account

            // Apply filters
            if ($request->has('start_date') && $request->start_date) {
                $query->whereDate('transaction_date', '>=', $request->start_date);
            }

            if ($request->has('end_date') && $request->end_date) {
                $query->whereDate('transaction_date', '<=', $request->end_date);
            }

            if ($request->has('transaction_type') && $request->transaction_type) {
                $query->where('transaction_type', $request->transaction_type);
            }

            if ($request->has('status') && $request->status) {
                $query->where('status', $request->status);
            }

            $transactions = $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20));

            // Transform transactions to use receiver_name as note value and fix date/time
            $transactionsArray = $transactions->toArray();
            
            // Fix each transaction in the data array
            foreach ($transactionsArray['data'] as &$trans) {
                // Use receiver_name for note if available
                $trans['note'] = $trans['receiver_name'] ?? $trans['note'];
                
                // Merge date and time from created_at into transaction_date as full datetime
                if (isset($trans['created_at'])) {
                    $createdAt = \Carbon\Carbon::parse($trans['created_at']);
                    // Store full datetime in transaction_date
                    $trans['transaction_date'] = $createdAt->toIso8601String();
                    $trans['transaction_time'] = $createdAt->format('H:i:s'); // Keep for backward compatibility
                }
            }
            
            // Calculate totals like in web interface
            $totalDeposit = collect($transactionsArray['data'])->where('transaction_type', 'deposit')
                                       ->sum('transaction_amount');
            $totalWithdrawal = collect($transactionsArray['data'])->where('transaction_type', 'withdrawal')
                                            ->sum('transaction_amount');

        return response()->json([
            'success' => true,
            'data' => $transactionsArray,
            'account' => [
                'id' => $account->id,
                'account_title' => $account->account_title,
                'balance' => $account->balance,
                'currency' => $account->currency
            ],
            'totals' => [
                'deposits' => $totalDeposit,
                'withdrawals' => $totalWithdrawal
            ]
        ], 200, [], JSON_UNESCAPED_UNICODE);
        }
        
        // If no account_id provided, get transactions for ALL user accounts
        $customer = $user->customer;
        
        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        // Get all customer's account IDs
        $customerAccountIds = $customer->accounts()->pluck('id');
        
        if ($customerAccountIds->isEmpty()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'current_page' => 1,
                    'data' => [],
                    'first_page_url' => null,
                    'from' => null,
                    'last_page' => 1,
                    'last_page_url' => null,
                    'links' => [],
                    'next_page_url' => null,
                    'path' => $request->url(),
                    'per_page' => $request->get('per_page', 20),
                    'prev_page_url' => null,
                    'to' => null,
                    'total' => 0
                ],
                'account' => null,
                'totals' => [
                    'deposits' => 0,
                    'withdrawals' => 0
                ]
            ]);
        }

        // Use the same logic as the Livewire TransactionsAccount component
        // This ensures we only return transactions that affect the customer
        $query = AccountTransaction::with([
            'from_account.customer:id,customer_name',
            'from_account.currency:id,currency_name',
            'to_account.customer:id,customer_name',
            'to_account.currency:id,currency_name',
            'fees:id,fee_amount,feeable_id,feeable_type'
        ])
        ->select([
            'id', 'transaction_type', 'transaction_amount', 
            DB::raw('DATE(transaction_date) as transaction_date'),
            'transaction_time', 'person_name', 'note', 'receiver_name', 'from_account_id', 
            'to_account_id', 'transaction_group_id', 'status', 'created_at', 'get_fee'
        ])
        ->where('transaction_type', '!=', 'fee') // Exclude fee transactions
        ->whereNotIn('from_account_id', [1]) // Exclude transactions FROM customer ID 1 (cashbox)
        ->whereIn('from_account_id', $customerAccountIds); // Only transactions FROM the customer's accounts

        // Apply filters
        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('transaction_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('transaction_date', '<=', $request->end_date);
        }

        if ($request->has('transaction_type') && $request->transaction_type) {
            $query->where('transaction_type', $request->transaction_type);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        $transactions = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));

        // Transform transactions to use receiver_name as note value and fix date/time
        $transactionsArray = $transactions->toArray();
        
        // Fix each transaction in the data array
        foreach ($transactionsArray['data'] as &$trans) {
            // Use receiver_name for note if available
            $trans['note'] = $trans['receiver_name'] ?? $trans['note'];
            
            // Merge date and time from created_at into transaction_date as full datetime
            if (isset($trans['created_at'])) {
                $createdAt = \Carbon\Carbon::parse($trans['created_at']);
                // Store full datetime in transaction_date
                $trans['transaction_date'] = $createdAt->toIso8601String();
                $trans['transaction_time'] = $createdAt->format('H:i:s'); // Keep for backward compatibility
            }
        }
        
        // Calculate totals like in web interface
        $totalDeposit = collect($transactionsArray['data'])->where('transaction_type', 'deposit')
                                   ->sum('transaction_amount');
        $totalWithdrawal = collect($transactionsArray['data'])->where('transaction_type', 'withdrawal')
                                        ->sum('transaction_amount');

        // Get the primary account (first account) for backward compatibility
        $primaryAccount = $customer->accounts()->with('currency')->first();

        return response()->json([
            'success' => true,
            'data' => $transactionsArray,
            'account' => $primaryAccount ? [
                'id' => $primaryAccount->id,
                'account_title' => $primaryAccount->account_title,
                'balance' => $primaryAccount->balance,
                'currency' => $primaryAccount->currency
            ] : null,
            'totals' => [
                'deposits' => $totalDeposit,
                'withdrawals' => $totalWithdrawal
            ]
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    /**
     * Create new transaction
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'from_account_id' => 'required|exists:accounts,id',
            'transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'transaction_amount' => 'required|numeric|min:0',
            'transaction_fee' => 'required|numeric|min:0',
            'get_fee' => 'required|numeric|min:0',
            'person_name' => 'nullable|string|max:255',
            'note' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        // Check if the account belongs to the customer
        $fromAccount = Account::where('id', $request->from_account_id)
            ->where('customer_id', $customer->id)
            ->first();

        if (!$fromAccount) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found or access denied'
            ], 403);
        }

        try {
            DB::beginTransaction();

            $transactionGroupId = Str::uuid();
            $currentDate = Carbon::now()->toDateString();
            $currentTime = Carbon::now()->toTimeString();

            // Get the cashbox account (where customer_id is 1) with matching currency
            $cashboxAccount = Account::where('customer_id', 1)
                ->where('currency_id', $fromAccount->currency_id)
                ->where('account_type', 'profit')
                ->first();

            if (!$cashboxAccount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cashbox account with matching currency not found'
                ], 400);
            }

            // Determine the to_account_id based on user role
            $userRole = $user->getRoleNames()->first();
            $toAccountId = $cashboxAccount->id; // Default to cashbox

            if (in_array($userRole, ['cashier', 'supervisor'])) {
                $toAccount = Account::where('customer_id', $user->customer->id)
                    ->where('currency_id', $fromAccount->currency_id)
                    ->first();
                
                if ($toAccount) {
                    $toAccountId = $toAccount->id;
                }
            }

            // Ensure proper UTF-8 encoding for Kurdish text before storing
            $personName = $request->person_name;
            $note = $request->note;

            // Create the main transaction
            $transactionFrom = AccountTransaction::create([
                'from_account_id' => $request->from_account_id,
                'to_account_id' => $toAccountId,
                'person_name' => $personName,
                'get_fee' => $request->get_fee,
                'transaction_type' => $request->transaction_type,
                'transaction_amount' => $request->transaction_amount,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $note,
                'user_id' => $user->id,
                'transaction_group_id' => $transactionGroupId,
                'status' => 'pending', // Mobile transactions start as pending
            ]);

            // Create corresponding transaction for the to_account
            $transactionTo = AccountTransaction::create([
                'from_account_id' => $toAccountId,
                'to_account_id' => $request->from_account_id,
                'person_name' => $personName,
                'get_fee' => $request->get_fee,
                'transaction_type' => $request->transaction_type,
                'transaction_amount' => $request->transaction_amount,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $note,
                'user_id' => $user->id,
                'transaction_group_id' => $transactionGroupId,
                'status' => 'pending',
            ]);

            // Create fees
            $transactionFrom->fees()->create([
                'fee_amount' => $request->get_fee,
                'fee_type' => $request->transaction_type . '_fee',
            ]);

            $transactionTo->fees()->create([
                'fee_amount' => $request->get_fee,
                'fee_type' => $request->transaction_type . '_fee',
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Transaction created successfully',
                'data' => [
                    'transaction_group_id' => $transactionGroupId,
                    'transaction_from' => $transactionFrom,
                    'transaction_to' => $transactionTo,
                ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Transaction creation failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get transaction details
     */
    public function show($id)
    {
        $user = Auth::user();
        $customer = $user->customer;

        $transaction = AccountTransaction::with([
            'from_account.customer:id,customer_name',
            'from_account.currency:id,currency_name',
            'to_account.customer:id,customer_name',
            'to_account.currency:id,currency_name',
            'fees:id,fee_amount,feeable_id,feeable_type',
            'files:id,file_path,fileable_id,fileable_type'
        ])
        ->select([
            'id', 'transaction_type', 'transaction_amount', 
            DB::raw('CAST(DATE(created_at) AS CHAR) as transaction_date'),
            'transaction_time', 'person_name', 'note', 'receiver_name', 'from_account_id', 
            'to_account_id', 'transaction_group_id', 'status', 'created_at', 'get_fee'
        ])
        ->find($id);

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found'
            ], 404);
        }

        // Check if user has access to this transaction
        $customerAccountIds = $customer->accounts()->pluck('id');
        if (!in_array($transaction->from_account_id, $customerAccountIds->toArray()) && 
            !in_array($transaction->to_account_id, $customerAccountIds->toArray())) {
            return response()->json([
                'success' => false,
                'message' => 'Access denied'
            ], 403);
        }

        // Transform transaction to array and fix date/time
        $transactionArray = $transaction->toArray();
        
        // Use receiver_name for note if available
        $transactionArray['note'] = $transactionArray['receiver_name'] ?? $transactionArray['note'];
        
        // Merge date and time from created_at into transaction_date as full datetime
        if (isset($transactionArray['created_at'])) {
            $createdAt = \Carbon\Carbon::parse($transactionArray['created_at']);
            // Store full datetime in transaction_date
            $transactionArray['transaction_date'] = $createdAt->toIso8601String();
            $transactionArray['transaction_time'] = $createdAt->format('H:i:s'); // Keep for backward compatibility
        }

        return response()->json([
            'success' => true,
            'data' => $transactionArray
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    /**
     * Export transactions to PDF
     */
    public function export(Request $request)
    {
        try {
            \Log::info('PDF Export called', [
                'user_id' => Auth::id(),
                'request_params' => $request->all()
            ]);
            
            $user = Auth::user();
            $customer = $user->customer;

            if (!$customer) {
                \Log::error('Customer profile not found for user: ' . $user->id);
                return response()->json([
                    'success' => false,
                    'message' => 'Customer profile not found'
                ], 404);
            }
            
            \Log::info('Customer found', ['customer_id' => $customer->id]);

            // Use the same logic as the Livewire TransactionsAccount component
            // This ensures we only return transactions that affect the customer
            $customerAccountIds = $customer->accounts()->pluck('id');
            
            $query = AccountTransaction::with([
                'from_account.customer:id,customer_name',
                'from_account.currency:id,currency_name',
                'to_account.customer:id,customer_name',
                'to_account.currency:id,currency_name',
                'fees:id,fee_amount,feeable_id,feeable_type'
            ])
            ->select([
                'id', 'transaction_type', 'transaction_amount', 
                DB::raw('CAST(DATE(created_at) AS CHAR) as transaction_date'),
                'transaction_time', 'person_name', 'note', 'receiver_name', 'from_account_id', 
                'to_account_id', 'transaction_group_id', 'status', 'created_at', 'get_fee'
            ])
            ->where('transaction_type', '!=', 'fee') // Exclude fee transactions
            ->whereNotIn('from_account_id', [1]) // Exclude transactions FROM customer ID 1 (cashbox)
            ->whereIn('from_account_id', $customerAccountIds); // Only transactions FROM the customer's accounts

            // Apply filters
            if ($request->has('start_date') && $request->start_date) {
                $query->whereDate('transaction_date', '>=', $request->start_date);
            }

            if ($request->has('end_date') && $request->end_date) {
                $query->whereDate('transaction_date', '<=', $request->end_date);
            }

            if ($request->has('transaction_type') && $request->transaction_type) {
                $query->where('transaction_type', $request->transaction_type);
            }

            if ($request->has('status') && $request->status) {
                $query->where('status', $request->status);
            }

            $transactions = $query->orderBy('created_at', 'desc')->get();
            
            // Transform transactions to use receiver_name as note value and fix date/time
            $transactions->transform(function ($transaction) {
                $transaction->note = $transaction->receiver_name ?? $transaction->note;
                
                // Merge date and time from created_at into transaction_date as full datetime
                $createdAt = $transaction->created_at;
                if ($createdAt instanceof \Carbon\Carbon) {
                    // Store full datetime in transaction_date
                    $transaction->transaction_date = $createdAt->toIso8601String();
                    // Format time as H:i:s only (no date) - keep for backward compatibility
                    $transaction->transaction_time = $createdAt->format('H:i:s');
                }
                
                return $transaction;
            });
            
            \Log::info('Transactions found', [
                'count' => $transactions->count(),
                'customer_account_ids' => $customerAccountIds->toArray()
            ]);

            // Generate PDF using DomPDF
            $pdf = app('dompdf.wrapper')->loadView('pdf.transactions', [
                'transactions' => $transactions,
                'customer' => $customer,
                'start_date' => $request->start_date ?? 'All',
                'end_date' => $request->end_date ?? 'All',
                'filters' => [
                    'transaction_type' => $request->transaction_type,
                    'status' => $request->status,
                ],
                'dark_mode' => $request->has('dark_mode') && $request->dark_mode === 'true'
            ]);

            $filename = 'transactions-' . ($request->start_date ?? 'all') . '-to-' . ($request->end_date ?? 'all') . '.pdf';

            \Log::info('PDF generated successfully', ['filename' => $filename]);
            return $pdf->download($filename);

        } catch (\Exception $e) {
            \Log::error('PDF generation failed', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error generating PDF: ' . $e->getMessage()
            ], 500);
        }
    }
}