<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AccountTransaction;
use App\Models\Account;
use App\Models\Currency;
use App\Models\CurrencyRateHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class StatisticsController extends Controller
{
    /**
     * Get transaction statistics for customer
     */
    public function transactionStats(Request $request)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        // Get all customer's account IDs
        $accountIds = $customer->accounts()->pluck('id')->toArray();

        if (empty($accountIds)) {
            return response()->json([
                'success' => true,
                'data' => $this->getEmptyStats()
            ]);
        }

        // Get overview statistics
        $overview = $this->getOverviewStats($accountIds);

        // Get monthly statistics (last 12 months)
        $monthlyStats = $this->getMonthlyStats($accountIds);

        // Get transaction types breakdown
        $transactionTypes = $this->getTransactionTypes($accountIds);

        return response()->json([
            'success' => true,
            'data' => [
                'overview' => $overview,
                'monthly_stats' => $monthlyStats,
                'transaction_types' => $transactionTypes,
            ]
        ]);
    }

    /**
     * Get overview statistics
     */
    private function getOverviewStats($accountIds)
    {
        $totalTransactions = AccountTransaction::where('transaction_type', '!=', 'fee')
            ->whereHas('from_account.customer', function ($query) {
                $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
            })
            ->where(function($q) use ($accountIds) {
                $q->whereIn('from_account_id', $accountIds)
                  ->orWhereIn('to_account_id', $accountIds);
            })
            ->count();

        $thisMonth = AccountTransaction::where('transaction_type', '!=', 'fee')
            ->whereHas('from_account.customer', function ($query) {
                $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
            })
            ->where(function($q) use ($accountIds) {
                $q->whereIn('from_account_id', $accountIds)
                  ->orWhereIn('to_account_id', $accountIds);
            })
            ->whereYear('transaction_date', now()->year)
            ->whereMonth('transaction_date', now()->month)
            ->count();

        $last30Days = AccountTransaction::where('transaction_type', '!=', 'fee')
            ->whereHas('from_account.customer', function ($query) {
                $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
            })
            ->where(function($q) use ($accountIds) {
                $q->whereIn('from_account_id', $accountIds)
                  ->orWhereIn('to_account_id', $accountIds);
            })
            ->where('transaction_date', '>=', now()->subDays(30))
            ->count();

        $activeCurrencies = Account::whereIn('id', $accountIds)
            ->distinct('currency_id')
            ->count('currency_id');

        return [
            'total_transactions' => $totalTransactions,
            'this_month' => $thisMonth,
            'last_30_days' => $last30Days,
            'active_currencies' => $activeCurrencies,
        ];
    }

    /**
     * Get monthly statistics for last 12 months
     */
    private function getMonthlyStats($accountIds)
    {
        $stats = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $month = $date->format('Y-m');
            
            $transactions = AccountTransaction::where('transaction_type', '!=', 'fee')
            ->whereHas('from_account.customer', function ($query) {
                $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
            })
            ->where(function($query) use ($accountIds) {
                $query->whereIn('from_account_id', $accountIds)
                      ->orWhereIn('to_account_id', $accountIds);
            })
            ->whereYear('transaction_date', $date->year)
            ->whereMonth('transaction_date', $date->month);

            $count = $transactions->count();
            $deposits = clone $transactions;
            $withdrawals = clone $transactions;

            $stats[] = [
                'month' => $month,
                'month_name' => $date->format('M'),
                'year' => $date->year,
                'count' => $count,
                'deposits' => $deposits->where('transaction_type', 'deposit')->count(),
                'withdrawals' => $withdrawals->where('transaction_type', 'withdrawal')->count(),
            ];
        }

        return $stats;
    }


    /**
     * Get transaction types breakdown
     */
    private function getTransactionTypes($accountIds)
    {
        $total = AccountTransaction::where('transaction_type', '!=', 'fee')
        ->whereHas('from_account.customer', function ($query) {
            $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
        })
        ->where(function($query) use ($accountIds) {
            $query->whereIn('from_account_id', $accountIds)
                  ->orWhereIn('to_account_id', $accountIds);
        })
        ->count();

        $deposits = AccountTransaction::where('transaction_type', 'deposit')
            ->whereHas('from_account.customer', function ($query) {
                $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
            })
            ->whereIn('to_account_id', $accountIds)
            ->count();

        $withdrawals = AccountTransaction::where('transaction_type', 'withdrawal')
            ->whereHas('from_account.customer', function ($query) {
                $query->where('id', '!=', 1); // Exclude customer ID 1 (cashbox)
            })
            ->whereIn('from_account_id', $accountIds)
            ->count();

        return [
            'deposits' => [
                'count' => $deposits,
                'percentage' => $total > 0 ? round(($deposits / $total) * 100, 1) : 0,
            ],
            'withdrawals' => [
                'count' => $withdrawals,
                'percentage' => $total > 0 ? round(($withdrawals / $total) * 100, 1) : 0,
            ],
        ];
    }

    /**
     * Get empty stats structure
     */
    private function getEmptyStats()
    {
        return [
            'overview' => [
                'total_transactions' => 0,
                'this_month' => 0,
                'last_30_days' => 0,
                'active_currencies' => 0,
            ],
            'monthly_stats' => [],
            'transaction_types' => [
                'deposits' => ['count' => 0, 'percentage' => 0],
                'withdrawals' => ['count' => 0, 'percentage' => 0],
            ],
        ];
    }

    /**
     * Get currency rate history
     */
    public function currencyRateHistory($currencyId, Request $request)
    {
        $days = $request->get('days', 30);
        $days = min($days, 365); // Max 1 year

        $currency = Currency::find($currencyId);

        if (!$currency) {
            return response()->json([
                'success' => false,
                'message' => 'Currency not found'
            ], 404);
        }

        $startDate = now()->subDays($days)->startOfDay();
        $endDate = now()->endOfDay();

        // Get history records
        $history = CurrencyRateHistory::forCurrency($currencyId)
            ->betweenDates($startDate, $endDate)
            ->orderBy('recorded_at', 'asc')
            ->get();

        // If no history, return current rate only
        if ($history->isEmpty()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'currency' => [
                        'id' => $currency->id,
                        'name' => $currency->currency_name,
                        'code' => $this->extractCurrencyCode($currency->currency_name),
                    ],
                    'current' => [
                        'buy_rate' => (float) $currency->buy_rate,
                        'sell_rate' => (float) $currency->sell_rate,
                        'last_updated' => now()->toISOString(),
                    ],
                    'history' => [],
                    'statistics' => [
                        'period_days' => $days,
                        'high' => (float) $currency->buy_rate,
                        'low' => (float) $currency->buy_rate,
                        'average' => (float) $currency->buy_rate,
                        'change_amount' => 0,
                        'change_percentage' => 0,
                        'trend' => 'stable',
                        'data_available' => false,
                    ],
                ]
            ]);
        }

        // Calculate statistics
        $buyRates = $history->pluck('buy_rate');
        $high = $buyRates->max();
        $low = $buyRates->min();
        $average = $buyRates->average();
        
        $firstRate = $history->first()->buy_rate;
        $lastRate = $history->last()->buy_rate;
        $changeAmount = $lastRate - $firstRate;
        $changePercentage = $firstRate > 0 ? (($changeAmount / $firstRate) * 100) : 0;
        
        $trend = 'stable';
        if ($changePercentage > 1) {
            $trend = 'up';
        } elseif ($changePercentage < -1) {
            $trend = 'down';
        }

        return response()->json([
            'success' => true,
            'data' => [
                'currency' => [
                    'id' => $currency->id,
                    'name' => $currency->currency_name,
                    'code' => $this->extractCurrencyCode($currency->currency_name),
                ],
                'current' => [
                    'buy_rate' => (float) $currency->buy_rate,
                    'sell_rate' => (float) $currency->sell_rate,
                    'last_updated' => now()->toISOString(),
                ],
                'history' => $history->map(function($record) {
                    return [
                        'date' => $record->recorded_at->format('Y-m-d'),
                        'buy_rate' => (float) $record->buy_rate,
                        'sell_rate' => (float) $record->sell_rate,
                        'timestamp' => $record->recorded_at->toISOString(),
                    ];
                })->values(),
                'statistics' => [
                    'period_days' => $days,
                    'high' => (float) $high,
                    'low' => (float) $low,
                    'average' => round($average, 2),
                    'change_amount' => round($changeAmount, 2),
                    'change_percentage' => round($changePercentage, 2),
                    'trend' => $trend,
                    'volatility' => $this->calculateVolatility($buyRates),
                    'data_available' => true,
                ],
            ]
        ]);
    }

    /**
     * Calculate volatility (simple method)
     */
    private function calculateVolatility($rates)
    {
        if ($rates->count() < 2) {
            return 'low';
        }

        $mean = $rates->average();
        $variance = $rates->map(function($rate) use ($mean) {
            return pow($rate - $mean, 2);
        })->average();
        
        $stdDev = sqrt($variance);
        $coefficientOfVariation = ($mean > 0) ? ($stdDev / $mean) * 100 : 0;

        if ($coefficientOfVariation < 1) {
            return 'low';
        } elseif ($coefficientOfVariation < 3) {
            return 'medium';
        } else {
            return 'high';
        }
    }

    /**
     * Extract currency code from currency name
     */
    private function extractCurrencyCode($currencyName)
    {
        // Known currency mappings
        $mappings = [
            'US Dollar' => 'USD',
            'Iraqi Dinar' => 'IQD',
            'Euro' => 'EUR',
            'British Pound' => 'GBP',
            'Japanese Yen' => 'JPY',
            'Canadian Dollar' => 'CAD',
            'Australian Dollar' => 'AUD',
            'Swiss Franc' => 'CHF',
            'Chinese Yuan' => 'CNY',
            'Turkish Lira' => 'TRY',
            'Saudi Riyal' => 'SAR',
            'UAE Dirham' => 'AED',
            'Kuwaiti Dinar' => 'KWD',
            'Bahraini Dinar' => 'BHD',
            'Qatari Riyal' => 'QAR',
            'Omani Rial' => 'OMR',
            'Jordanian Dinar' => 'JOD',
            'Lebanese Pound' => 'LBP',
            'Syrian Pound' => 'SYP',
            'Iranian Rial' => 'IRR',
            'Afghan Afghani' => 'AFN',
            'Pakistani Rupee' => 'PKR',
            'Indian Rupee' => 'INR',
            'Bangladeshi Taka' => 'BDT',
            'Sri Lankan Rupee' => 'LKR',
            'Nepalese Rupee' => 'NPR',
            'Myanmar Kyat' => 'MMK',
            'Thai Baht' => 'THB',
            'Vietnamese Dong' => 'VND',
            'Indonesian Rupiah' => 'IDR',
            'Malaysian Ringgit' => 'MYR',
            'Singapore Dollar' => 'SGD',
            'Philippine Peso' => 'PHP',
            'South Korean Won' => 'KRW',
            'New Zealand Dollar' => 'NZD',
            'South African Rand' => 'ZAR',
            'Egyptian Pound' => 'EGP',
            'Moroccan Dirham' => 'MAD',
            'Tunisian Dinar' => 'TND',
            'Algerian Dinar' => 'DZD',
            'Libyan Dinar' => 'LYD',
            'Sudanese Pound' => 'SDG',
            'Ethiopian Birr' => 'ETB',
            'Kenyan Shilling' => 'KES',
            'Ugandan Shilling' => 'UGX',
            'Tanzanian Shilling' => 'TZS',
            'Nigerian Naira' => 'NGN',
            'Ghanaian Cedi' => 'GHS',
            'West African CFA Franc' => 'XOF',
            'Central African CFA Franc' => 'XAF',
            'Brazilian Real' => 'BRL',
            'Argentine Peso' => 'ARS',
            'Chilean Peso' => 'CLP',
            'Colombian Peso' => 'COP',
            'Peruvian Sol' => 'PEN',
            'Mexican Peso' => 'MXN',
            'Venezuelan Bolívar' => 'VES',
            'Russian Ruble' => 'RUB',
            'Ukrainian Hryvnia' => 'UAH',
            'Polish Zloty' => 'PLN',
            'Czech Koruna' => 'CZK',
            'Hungarian Forint' => 'HUF',
            'Romanian Leu' => 'RON',
            'Bulgarian Lev' => 'BGN',
            'Croatian Kuna' => 'HRK',
            'Serbian Dinar' => 'RSD',
            'Albanian Lek' => 'ALL',
            'Macedonian Denar' => 'MKD',
            'Bosnian Convertible Mark' => 'BAM',
            'Montenegrin Euro' => 'EUR',
            'Kosovo Euro' => 'EUR',
        ];

        // Check if we have a direct mapping
        if (isset($mappings[$currencyName])) {
            return $mappings[$currencyName];
        }

        // Try to extract from currency name (take first 3 characters of each word)
        $words = explode(' ', $currencyName);
        if (count($words) >= 2) {
            // For compound names like "US Dollar", take first 2 letters from each word
            $code = '';
            foreach (array_slice($words, 0, 2) as $word) {
                $code .= strtoupper(substr($word, 0, 2));
            }
            return $code;
        }

        // Fallback: take first 3 characters
        return strtoupper(substr($currencyName, 0, 3));
    }

    /**
     * Record current currency rates (called by scheduler)
     */
    public function recordCurrentRates()
    {
        $currencies = Currency::all();
        $recorded = 0;

        foreach ($currencies as $currency) {
            // Only record if rates exist
            if ($currency->buy_rate && $currency->sell_rate) {
                CurrencyRateHistory::create([
                    'currency_id' => $currency->id,
                    'buy_rate' => $currency->buy_rate,
                    'sell_rate' => $currency->sell_rate,
                    'recorded_at' => now(),
                ]);
                $recorded++;
            }
        }

        \Log::info("Recorded {$recorded} currency rates");

        return [
            'success' => true,
            'recorded' => $recorded,
            'timestamp' => now()->toISOString(),
        ];
    }
}

