<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Services\NotificationTranslationService;

class OneSignalController extends Controller
{
    /**
     * Update OneSignal Subscription ID for authenticated user (new method)
     */
    public function updateSubscriptionId(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'subscription_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $user = Auth::user();
            $user->onesignal_player_id = $request->subscription_id; // Store in same field for now
            $user->save();

            return response()->json([
                'success' => true,
                'message' => 'OneSignal Subscription ID updated successfully',
                'data' => [
                    'user_id' => $user->id,
                    'subscription_id' => $user->onesignal_player_id,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update OneSignal Subscription ID',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update OneSignal player ID for authenticated user (deprecated - use updateSubscriptionId)
     */
    public function updatePlayerId(Request $request): JsonResponse
    {
        \Log::info('OneSignal updatePlayerId called', [
            'user_id' => Auth::id(),
            'request_data' => $request->all(),
            'headers' => $request->headers->all()
        ]);

        $validator = Validator::make($request->all(), [
            'player_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            \Log::error('OneSignal updatePlayerId validation failed', [
                'user_id' => Auth::id(),
                'errors' => $validator->errors()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $user = Auth::user();
            \Log::info('OneSignal updatePlayerId - user found', [
                'user_id' => $user->id,
                'current_player_id' => $user->onesignal_player_id,
                'new_player_id' => $request->player_id
            ]);

            $user->onesignal_player_id = $request->player_id;
            $user->save();

            \Log::info('OneSignal updatePlayerId - saved successfully', [
                'user_id' => $user->id,
                'saved_player_id' => $user->onesignal_player_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'OneSignal player ID updated successfully',
                'data' => [
                    'user_id' => $user->id,
                    'player_id' => $user->onesignal_player_id,
                ],
            ]);

        } catch (\Exception $e) {
            \Log::error('OneSignal updatePlayerId - exception', [
                'user_id' => Auth::id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to update OneSignal player ID',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get OneSignal player ID for authenticated user
     */
    public function getPlayerId(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();

            return response()->json([
                'success' => true,
                'data' => [
                    'user_id' => $user->id,
                    'player_id' => $user->onesignal_player_id,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get OneSignal player ID',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update OneSignal External ID for authenticated user
     */
    public function updateExternalId(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'external_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $user = Auth::user();
            $user->onesignal_external_id = $request->external_id;
            $user->save();

            \Log::info('OneSignal External ID updated', [
                'user_id' => $user->id,
                'external_id' => $user->onesignal_external_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'OneSignal External ID updated successfully',
                'data' => [
                    'user_id' => $user->id,
                    'external_id' => $user->onesignal_external_id,
                ],
            ]);

        } catch (\Exception $e) {
            \Log::error('OneSignal External ID update failed', [
                'user_id' => Auth::id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update OneSignal External ID',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Test notification for authenticated user
     */
    public function testNotification(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();

            // Use user ID directly as External ID (they are the same in OneSignal)
            $userId = $user->id;

            // Send test notification using user ID as External ID
            $oneSignalService = app(\App\Services\OneSignalService::class);
            $translationService = new NotificationTranslationService();
            $testMessage = $translationService->getTestMessage();
            
            $result = $oneSignalService->sendNotificationToUsers(
                [$userId], // Use user ID directly as External ID
                $testMessage,
                [
                    'type' => 'test',
                    'timestamp' => now()->toISOString(),
                    'user_id' => $userId,
                    'external_id' => $userId // User ID and External ID are the same
                ]
            );

            // Handle the new detailed response format
            if (is_array($result)) {
                if ($result['success'] === true) {
                    return response()->json([
                        'success' => true,
                        'message' => 'تەستکردنی ئاگادارکردنەوە بە سەرکەوتوویی نێردرا!',
                        'data' => [
                            'user_id' => $userId,
                            'external_id' => $userId, // User ID and External ID are the same
                            'recipients' => $result['recipients'] ?? 1,
                            'onesignal_response' => $result['data'] ?? null,
                            'message_sent' => $testMessage
                        ]
                    ]);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => $result['error'] ?? 'Failed to send test notification',
                        'error_type' => $result['error_type'] ?? 'unknown',
                        'details' => $result,
                        'user_id' => $userId,
                        'external_id' => $userId, // User ID and External ID are the same
                    ], 500);
                }
            } else {
                // Fallback for old boolean response format
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to send test notification - service returned unexpected response format',
                    'error_type' => 'service_error',
                    'user_id' => $user->id,
                    'player_id' => $user->onesignal_player_id,
                ], 500);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unexpected error occurred while sending test notification',
                'error_type' => 'exception',
                'error' => $e->getMessage(),
                'user_id' => Auth::id(),
            ], 500);
        }
    }
}


