<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class NotificationController extends Controller
{
    /**
     * Get all notifications for the authenticated user.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();
            $notifications = Notification::forUser($user->id)
                ->orderBy('created_at', 'desc')
                ->get();

            // Transform notifications to match mobile app format
            $formattedNotifications = $notifications->map(function ($notification) {
                return [
                    'id' => $notification->id,
                    'title' => $notification->title,
                    'body' => $notification->body,
                    'image_url' => $notification->image_url,
                    'data' => $notification->data,
                    'action_url' => $notification->action_url,
                    'is_read' => $notification->is_read,
                    'created_at' => $notification->created_at->toISOString(),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $formattedNotifications,
                'total' => $notifications->count(),
                'unread_count' => $notifications->where('is_read', false)->count(),
            ], 200, [
                'Content-Type' => 'application/json; charset=utf-8',
                'Content-Encoding' => 'utf-8'
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch notifications',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get a specific notification.
     */
    public function show(int $id): JsonResponse
    {
        try {
            $user = Auth::user();
            $notification = Notification::forUser($user->id)->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $notification->id,
                    'title' => $notification->title,
                    'body' => $notification->body,
                    'image_url' => $notification->image_url,
                    'data' => $notification->data,
                    'action_url' => $notification->action_url,
                    'is_read' => $notification->is_read,
                    'created_at' => $notification->created_at->toISOString(),
                ],
            ], 200, [
                'Content-Type' => 'application/json; charset=utf-8',
                'Content-Encoding' => 'utf-8'
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Notification not found',
                'error' => $e->getMessage(),
            ], 404);
        }
    }

    /**
     * Mark a notification as read.
     */
    public function markAsRead(int $id): JsonResponse
    {
        try {
            $user = Auth::user();
            $notification = Notification::forUser($user->id)->findOrFail($id);

            if (!$notification->is_read) {
                $notification->markAsRead();
            }

            return response()->json([
                'success' => true,
                'message' => 'Notification marked as read',
                'data' => [
                    'id' => $notification->id,
                    'is_read' => $notification->is_read,
                    'read_at' => $notification->read_at?->toISOString(),
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark notification as read',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Mark all notifications as read.
     */
    public function markAllAsRead(): JsonResponse
    {
        try {
            $user = Auth::user();
            $updatedCount = Notification::forUser($user->id)
                ->unread()
                ->update([
                    'is_read' => true,
                    'read_at' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'All notifications marked as read',
                'data' => [
                    'updated_count' => $updatedCount,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark all notifications as read',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete a notification.
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $user = Auth::user();
            $notification = Notification::forUser($user->id)->findOrFail($id);
            $notification->delete();

            return response()->json([
                'success' => true,
                'message' => 'Notification deleted successfully',
                'data' => [
                    'id' => $id,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete notification',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Clear all notifications.
     */
    public function clearAll(): JsonResponse
    {
        try {
            $user = Auth::user();
            $deletedCount = Notification::forUser($user->id)->delete();

            return response()->json([
                'success' => true,
                'message' => 'All notifications cleared',
                'data' => [
                    'deleted_count' => $deletedCount,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear all notifications',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get unread notifications count.
     */
    public function unreadCount(): JsonResponse
    {
        try {
            $user = Auth::user();
            $unreadCount = Notification::forUser($user->id)->unread()->count();

            return response()->json([
                'success' => true,
                'count' => $unreadCount,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get unread count',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Create a notification (for testing or admin use).
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'type' => 'required|string|in:app,transaction,currency_rate,currency_rate_update,security',
                'title' => 'required|string|max:255',
                'body' => 'required|string',
                'image_url' => 'nullable|url',
                'data' => 'nullable|array',
                'action_url' => 'nullable|string',
                'user_id' => 'nullable|exists:users,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $user = Auth::user();
            $targetUserId = $request->input('user_id', $user->id);

            $notification = Notification::create([
                'type' => $request->input('type'),
                'title' => $request->input('title'),
                'body' => $request->input('body'),
                'image_url' => $request->input('image_url'),
                'data' => $request->input('data'),
                'action_url' => $request->input('action_url'),
                'user_id' => $targetUserId,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Notification created successfully',
                'data' => [
                    'id' => $notification->id,
                    'title' => $notification->title,
                    'body' => $notification->body,
                    'created_at' => $notification->created_at->toISOString(),
                ],
            ], 201, [
                'Content-Type' => 'application/json; charset=utf-8',
                'Content-Encoding' => 'utf-8'
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create notification',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
