<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Models\AccountTransaction;
use App\Http\Controllers\Controller;
use App\Http\Resources\AccountTransactionResource;
use App\Http\Resources\AccountTransactionCollection;
use App\Http\Requests\AccountTransactionStoreRequest;
use App\Http\Requests\AccountTransactionUpdateRequest;
use App\Services\TransactionNotificationService;

class AccountTransactionController extends Controller
{
    protected $notificationService;

    public function __construct(TransactionNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    public function index(Request $request): AccountTransactionCollection
    {
        $this->authorize('view-any', AccountTransaction::class);

        $search = $request->get('search', '');

        $accountTransactions = AccountTransaction::search($search)
            ->latest()
            ->paginate();

        return new AccountTransactionCollection($accountTransactions);
    }

    public function store(
        AccountTransactionStoreRequest $request
    ): AccountTransactionResource {
        $this->authorize('create', AccountTransaction::class);

        $validated = $request->validated();

        $accountTransaction = AccountTransaction::create($validated);

        // Send notification for new transaction
        $this->notificationService->sendTransactionCreatedNotification($accountTransaction);

        return new AccountTransactionResource($accountTransaction);
    }

    public function show(
        Request $request,
        AccountTransaction $accountTransaction
    ): AccountTransactionResource {
        $this->authorize('view', $accountTransaction);

        return new AccountTransactionResource($accountTransaction);
    }

    public function update(
        AccountTransactionUpdateRequest $request,
        AccountTransaction $accountTransaction
    ): AccountTransactionResource {
        $this->authorize('update', $accountTransaction);

        $validated = $request->validated();
        $oldStatus = $accountTransaction->status;

        $accountTransaction->update($validated);

        // Send notification if status changed
        if (isset($validated['status']) && $validated['status'] !== $oldStatus) {
            switch ($validated['status']) {
                case 'approved':
                    $this->notificationService->sendTransactionApprovedNotification($accountTransaction);
                    break;
                case 'rejected':
                    $this->notificationService->sendTransactionRejectedNotification($accountTransaction);
                    break;
            }
        }

        return new AccountTransactionResource($accountTransaction);
    }

    public function destroy(
        Request $request,
        AccountTransaction $accountTransaction
    ): Response {
        $this->authorize('delete', $accountTransaction);

        $accountTransaction->delete();

        return response()->noContent();
    }
}
