<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Customer;

class TestAccountsApi extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:accounts-api';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test the accounts API response structure';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== TESTING ACCOUNTS API ===');

        try {
            // Get a user to test with
            $user = User::whereHas('customer')->first();
            
            if (!$user) {
                $this->error('No users found with customer relationship');
                return;
            }
            
            $this->info("Found user: {$user->email}");
            $this->info("Customer ID: {$user->customer->id}");
            
            // Simulate the accounts API call
            $customer = $user->customer;
            $accounts = $customer->accounts()->with('currency:id,currency_name,code')->get();
            
            $this->info("Found {$accounts->count()} accounts for customer");
            
            // Group accounts by currency and calculate total balance for each currency
            $currencyBalances = $accounts->groupBy('currency_id')->map(function ($currencyAccounts) {
                $currency = $currencyAccounts->first()->currency;
                $totalBalance = $currencyAccounts->sum(function ($account) {
                    return $account->calculateTotalBalance();
                });

                return [
                    'id' => $currency->id,
                    'currency_name' => $currency->currency_name,
                    'currency_code' => $currency->code ?? 'USD',
                    'total_balance' => $totalBalance,
                    'account_count' => $currencyAccounts->count()
                ];
            });
            
            $this->info('Currency balances calculated:');
            foreach ($currencyBalances as $balance) {
                $this->line("   - {$balance['currency_name']}: {$balance['total_balance']} ({$balance['account_count']} accounts)");
            }
            
            $this->info('Final API response structure:');
            $response = [
                'success' => true,
                'data' => $currencyBalances->values()->toArray()
            ];
            
            $this->line(json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        } catch (\Exception $e) {
            $this->error("Error: " . $e->getMessage());
            $this->line("Stack trace: " . $e->getTraceAsString());
        }

        $this->info('=== TEST COMPLETE ===');
    }
}
