<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Customer;
use App\Models\Account;

class TestAccountStatement extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:account-statement';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test account statement API endpoint';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== TESTING ACCOUNT STATEMENT API ===');

        try {
            // Get user ID 1 (admin@admin.com)
            $user = User::find(1);
            
            if (!$user) {
                $this->error('User ID 1 not found');
                return;
            }
            
            $this->info("Testing with user: {$user->email} (ID: {$user->id})");
            
            // Check customer relationship
            $customer = $user->customer;
            if (!$customer) {
                $this->error('No customer found for user');
                return;
            }
            
            $this->info("Customer: {$customer->customer_name} (ID: {$customer->id})");
            
            // Test the exact logic from CustomerController
            $this->info("\n=== SIMULATING CUSTOMER CONTROLLER LOGIC ===");
            
            // Simulate Auth::user()
            $authenticatedUser = $user;
            $authenticatedCustomer = $authenticatedUser->customer;
            
            if (!$authenticatedCustomer) {
                $this->error('❌ Customer profile not found - would return 404');
                return;
            }
            
            $this->info('✅ Customer profile found');
            
            // Test account lookup for account ID 2
            $accountId = 2;
            $startDate = '2025-02-01';
            $endDate = '2025-10-01';
            
            $this->info("Testing account ID: $accountId");
            $this->info("Date range: $startDate to $endDate");
            
            // Find the specific account (exact logic from getDetailedAccountStatement)
            $account = $authenticatedCustomer->accounts()->with('currency:id,currency_name')->find($accountId);
            
            if (!$account) {
                $this->error("❌ Account not found or does not belong to customer - would return 404");
                $this->line("This means the account ID $accountId either:");
                $this->line("1. Doesn't exist");
                $this->line("2. Doesn't belong to customer ID {$authenticatedCustomer->id}");
                
                // Let's check what accounts this customer actually has
                $customerAccounts = $authenticatedCustomer->accounts()->with('currency:id,currency_name')->get();
                $this->info("\nCustomer {$authenticatedCustomer->id} actually has these accounts:");
                foreach ($customerAccounts->take(10) as $acc) {
                    $this->line("  - Account ID: {$acc->id}, Currency: " . ($acc->currency ? $acc->currency->currency_name : 'NULL'));
                }
                if ($customerAccounts->count() > 10) {
                    $this->line("  ... and " . ($customerAccounts->count() - 10) . " more accounts");
                }
                
                return;
            }
            
            $this->info("✅ Account found: {$account->account_title} (Currency: " . ($account->currency ? $account->currency->currency_name : 'NULL') . ")");
            
            // Test the rest of the logic
            $this->info("✅ Account statement would be generated successfully");
            
        } catch (\Exception $e) {
            $this->error("Error: " . $e->getMessage());
            $this->line("Stack trace: " . $e->getTraceAsString());
        }

        $this->info('=== TEST COMPLETE ===');
    }
}
