<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use ZipArchive;

class CreateBackup extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'backup:create';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a database backup';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        try {
            // Create a backup filename with timestamp
            $timestamp = Carbon::now()->format('Y-m-d-H-i-s');
            $filename = "backup-{$timestamp}.sql";
            $backupPath = storage_path("app/backups/{$filename}");
            
            // Ensure the backups directory exists
            if (!file_exists(dirname($backupPath))) {
                mkdir(dirname($backupPath), 0755, true);
            }
            
            // Open the SQL file for writing
            $file = fopen($backupPath, 'w');
            
            // Add SQL header with timestamp
            fwrite($file, "-- Database backup created on " . date('Y-m-d H:i:s') . "\n");
            fwrite($file, "-- ------------------------------------------------------------\n\n");
            
            // Get all tables
            $tables = DB::select('SHOW TABLES');
            
            foreach ($tables as $table) {
                $tableName = get_object_vars($table)[key(get_object_vars($table))];
                
                // Add table structure
                fwrite($file, "-- Table structure for table `{$tableName}`\n");
                fwrite($file, "DROP TABLE IF EXISTS `{$tableName}`;\n");
                
                // Get create table statement
                $createTableSql = DB::select("SHOW CREATE TABLE `{$tableName}`");
                $createTableStatement = get_object_vars($createTableSql[0])['Create Table'] ?? null;
                
                if ($createTableStatement) {
                    fwrite($file, $createTableStatement . ";\n\n");
                }
                
                // Get table data
                $rows = DB::table($tableName)->get();
                
                if (count($rows) > 0) {
                    fwrite($file, "-- Dumping data for table `{$tableName}`\n");
                    
                    // Get column names
                    $columns = Schema::getColumnListing($tableName);
                    
                    // Create INSERT statements in batches
                    $batchSize = 100;
                    $totalRows = count($rows);
                    
                    for ($i = 0; $i < $totalRows; $i += $batchSize) {
                        $batchRows = array_slice($rows->toArray(), $i, $batchSize);
                        
                        if (empty($batchRows)) {
                            continue;
                        }
                        
                        fwrite($file, "INSERT INTO `{$tableName}` (`" . implode("`, `", $columns) . "`) VALUES\n");
                        
                        $valueStrings = [];
                        foreach ($batchRows as $row) {
                            $values = [];
                            $rowArray = (array) $row;
                            
                            foreach ($columns as $column) {
                                $value = $rowArray[$column] ?? null;
                                
                                if (is_null($value)) {
                                    $values[] = "NULL";
                                } elseif (is_numeric($value)) {
                                    $values[] = $value;
                                } else {
                                    $values[] = "'" . str_replace("'", "''", $value) . "'";
                                }
                            }
                            
                            $valueStrings[] = "(" . implode(", ", $values) . ")";
                        }
                        
                        fwrite($file, implode(",\n", $valueStrings) . ";\n\n");
                    }
                }
            }
            
            // Close the file
            fclose($file);
            
            // Compress the SQL file to ZIP
            $zipFilename = "backup-{$timestamp}.zip";
            $zipPath = storage_path("app/backups/{$zipFilename}");
            
            $zip = new ZipArchive();
            if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
                $zip->addFile($backupPath, basename($backupPath));
                $zip->close();
                
                // Remove the SQL file if ZIP was created successfully
                if (file_exists($zipPath)) {
                    unlink($backupPath);
                }
                
                $this->info('Database backup created successfully');
                return 0;
            }
            
            // If ZIP creation fails, keep the SQL file
            $this->info('Database backup created successfully (uncompressed)');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('Backup failed: ' . $e->getMessage());
            return 1;
        }
    }
}