<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Customer;

class CheckAuth extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'check:auth';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check user authentication and customer relationship';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== CHECKING AUTHENTICATION & CUSTOMER RELATIONSHIP ===');

        try {
            // Check all users and their customer relationships
            $users = User::with('customer')->get();
            
            $this->info("Found {$users->count()} users:");
            
            foreach ($users as $user) {
                $this->line("User ID: {$user->id}, Email: {$user->email}");
                
                if ($user->customer) {
                    $this->info("  ✅ Has customer: ID {$user->customer->id}, Name: {$user->customer->customer_name}");
                    
                    // Check accounts for this customer
                    $accounts = $user->customer->accounts;
                    $this->line("  📊 Has {$accounts->count()} accounts");
                    
                    if ($accounts->count() > 0) {
                        foreach ($accounts->take(3) as $account) {
                            $this->line("     - Account ID: {$account->id}, Currency: " . ($account->currency ? $account->currency->currency_name : 'NULL'));
                        }
                        if ($accounts->count() > 3) {
                            $this->line("     ... and " . ($accounts->count() - 3) . " more");
                        }
                    }
                } else {
                    $this->warn("  ⚠️ No customer record found");
                }
                $this->line("");
            }
            
            // Check if there are customers without users
            $this->info("=== CUSTOMERS WITHOUT USERS ===");
            $customersWithoutUsers = Customer::whereNull('user_id')->orWhereNotIn('user_id', $users->pluck('id'))->get();
            $this->line("Found {$customersWithoutUsers->count()} customers without valid user relationships");
            
            foreach ($customersWithoutUsers->take(5) as $customer) {
                $this->line("Customer ID: {$customer->id}, Name: {$customer->customer_name}, User ID: {$customer->user_id}");
            }

        } catch (\Exception $e) {
            $this->error("Error: " . $e->getMessage());
            $this->line("Stack trace: " . $e->getTraceAsString());
        }

        $this->info('=== CHECK COMPLETE ===');
    }
}
